import type { FormatMessage } from 'tachyon-intl';
import { exhaustedCase } from 'tachyon-utils-ts';
import type {
  ReportContentErrorCode,
  ReportContentType,
  ReportWizardFragment,
  ReportableContentFragment,
} from './gql-types';
import type {
  GracefulContentType,
  GracefulReportWizardFragment,
} from './models';
import { WizardEvent } from './tracking';
import type { TrackWizardEvent } from './tracking';

// istanbul ignore next: trivial
export const getContentString = (
  content: ReportContentType | undefined,
  formatMessage: FormatMessage,
): string => {
  if (!content) {
    return '';
  }
  switch (content) {
    case 'CHANNEL_FEED_COMMENT_REPORT':
      return formatMessage('Channel Feed Comment', 'ReportWizardForm');
    case 'CHANNEL_FEED_POST_REPORT':
      return formatMessage('Channel Feed Post', 'ReportWizardForm');
    case 'CHANNEL_POINTS_REPORT':
      return formatMessage('Channel Points', 'ReportWizardForm');
    case 'CHAT_REPORT':
      return formatMessage('Chat Messages', 'ReportWizardForm');
    case 'CLIP_REPORT':
      return formatMessage('Clip', 'ReportWizardForm');
    case 'COLLECTION_REPORT':
      return formatMessage('Collection', 'ReportWizardForm');
    case 'CREATOR_GOAL_REPORT':
      return formatMessage('Creator Goal', 'ReportWizardForm');
    case 'CURSE_PRIVATE_GROUP':
      return formatMessage('Curse Private Group', 'ReportWizardForm');
    case 'CURSE_PUBLIC_GROUP':
      return formatMessage('Curse Public Group', 'ReportWizardForm');
    case 'CURSE_WHISPER':
      return formatMessage('Curse Whisper', 'ReportWizardForm');
    case 'EMOTE_REPORT':
      return formatMessage('Emote', 'ReportWizardForm');
    case 'EVENT_REPORT':
      return formatMessage('Event', 'ReportWizardForm');
    case 'EXTENSION_REPORT':
      return formatMessage('Extension', 'ReportWizardForm');
    case 'LIVE_UP_REPORT':
      return formatMessage('Live Up', 'ReportWizardForm');
    case 'LIVESTREAM_REPORT':
      return formatMessage('Live Stream', 'ReportWizardForm');
    case 'OFF_PLATFORM_REPORT':
      return formatMessage('Off Twitch Behavior', 'ReportWizardForm');
    case 'POLL_REPORT':
      return formatMessage('Poll', 'ReportWizardForm');
    case 'RAID_REPORT':
      return formatMessage('Raid', 'ReportWizardForm');
    case 'REWARD_REDEMPTION_REPORT':
      return formatMessage('Reward Redemption', 'ReportWizardForm');
    case 'ROOM_REPORT':
      return formatMessage('Room', 'ReportWizardForm');
    case 'SINGS_DUET_SEED_REPORT':
      return formatMessage('Sings Duet Seed', 'ReportWizardForm');
    case 'SINGS_GROUP_BIO_REPORT':
      return formatMessage('Sings Group Bio', 'ReportWizardForm');
    case 'SINGS_GROUP_CHAT_REPORT':
      return formatMessage('Sings Group Chat', 'ReportWizardForm');
    case 'SINGS_GROUP_NAME_REPORT':
      return formatMessage('Sings Group Name', 'ReportWizardForm');
    case 'SINGS_VOD_COMMENT_REPORT':
      return formatMessage('Sings VOD Comment', 'ReportWizardForm');
    case 'UNBAN_REQUEST_REPORT':
      return formatMessage('Unban Request', 'ReportWizardForm');
    case 'USER_REPORT':
      return formatMessage('User', 'ReportWizardForm');
    case 'USERNAME_REPORT':
      return formatMessage('Username', 'ReportWizardForm');
    case 'VOD_COMMENT_REPORT':
      return formatMessage('VOD Comment', 'ReportWizardForm');
    case 'VOD_REPORT':
      return formatMessage('VOD or Highlight', 'ReportWizardForm');
    case 'WHISPER_REPORT':
      return formatMessage('Whispers', 'ReportWizardForm');
    default:
      return exhaustedCase(content, '');
  }
};

// istanbul ignore next: trivial
export const getReportUserContentErrorString = (
  code: ReportContentErrorCode,
  formatMessage: FormatMessage,
): string => {
  switch (code) {
    case 'INTERNAL_ERROR':
      return formatMessage('Internal error', 'ReportWizardForm');
    case 'INVALID_DESCRIPTION_TOO_LONG':
      return formatMessage('Description is too long', 'ReportWizardForm');
    case 'INVALID_EMAIL':
      return formatMessage('Email is invalid', 'ReportWizardForm');
    case 'INVALID_NO_WHISPERS_WITH_TARGET':
      return formatMessage(
        'Cannot find whispers between you and reported user',
        'ReportWizardForm',
      );
    default:
      return exhaustedCase(code, '');
  }
};

// istanbul ignore next: trivial
export const toValidReportContentType = (
  content: GracefulContentType,
  onEvent?: TrackWizardEvent,
): ReportContentType => {
  switch (content) {
    case 'CHANNEL_FEED_COMMENT_REPORT':
      return content;
    case 'CHANNEL_FEED_POST_REPORT':
      return content;
    case 'CHANNEL_POINTS_REPORT':
      return content;
    case 'CHAT_REPORT':
      return content;
    case 'CLIP_REPORT':
      return content;
    case 'COLLECTION_REPORT':
      return content;
    case 'CREATOR_GOAL_REPORT':
      return content;
    case 'CURSE_PRIVATE_GROUP':
      return content;
    case 'CURSE_PUBLIC_GROUP':
      return content;
    case 'CURSE_WHISPER':
      return content;
    case 'EMOTE_REPORT':
      return content;
    case 'EVENT_REPORT':
      return content;
    case 'EXTENSION_REPORT':
      return content;
    case 'LIVE_UP_REPORT':
      return content;
    case 'LIVESTREAM_REPORT':
      return content;
    case 'OFF_PLATFORM_REPORT':
      return content;
    case 'POLL_REPORT':
      return content;
    case 'RAID_REPORT':
      return content;
    case 'REWARD_REDEMPTION_REPORT':
      return content;
    case 'ROOM_REPORT':
      return content;
    case 'SINGS_DUET_SEED_REPORT':
      return content;
    case 'SINGS_GROUP_BIO_REPORT':
      return content;
    case 'SINGS_GROUP_CHAT_REPORT':
      return content;
    case 'SINGS_GROUP_NAME_REPORT':
      return content;
    case 'SINGS_VOD_COMMENT_REPORT':
      return content;
    case 'UNBAN_REQUEST_REPORT':
      return content;
    case 'USER_REPORT':
      return content;
    case 'USERNAME_REPORT':
      return content;
    case 'VOD_COMMENT_REPORT':
      return content;
    case 'VOD_REPORT':
      return content;
    case 'WHISPER_REPORT':
      return content;
    default:
      /* if an unexpected content type is passed in, we default to USER_REPORT as a fallback
      the reason for this is that we don't want twilight builds to fail once we've merged
      a new content type to graphQL's main branch but have not cut a new tachyon release with
      updated generated types yet. also call onEvent so report wizard package consumers have
      the ability to configure an alarm via spud if events are handled with spade */

      onEvent?.(WizardEvent.ReportWizardUnknownContentType, {
        unknown_content_type_value: content,
      });
      return 'USER_REPORT';
  }
};

export const validatedData = (
  data: GracefulReportWizardFragment,
): ReportWizardFragment => {
  const reportableContent: ReportableContentFragment[] =
    data.reportableContent.map((content) => {
      return {
        applicableReasons: content.applicableReasons,
        deadEndType: content.deadEndType,
        id: content.id,
        type: toValidReportContentType(content.type),
      };
    });

  return {
    reasons: data.reasons,
    reportableContent,
  };
};
