import type { Request, Response } from 'express';
import type { CurrentUser } from 'tachyon-auth';
import { TACHYON_USER_COOKIE_NAME } from 'tachyon-auth';
import {
  GLOBAL_COOKIE_NAMES,
  clearCookieValueOnServer,
  getAndExtendCookieValueOnServer,
} from 'tachyon-server-utils';

function getAuthTokenOnServer(req: Request, res: Response): string | null {
  return (
    getAndExtendCookieValueOnServer({
      name: GLOBAL_COOKIE_NAMES.AUTH_TOKEN_COOKIE_NAME,
      req,
      res,
    }) ?? null
  );
}

/**
 * Performs a server side logout. Clears the auth token cookie and the user cookie.
 */
export function logoutOnServer(res: Response): void {
  clearCookieValueOnServer({
    name: GLOBAL_COOKIE_NAMES.AUTH_TOKEN_COOKIE_NAME,
    res,
  });
  clearCookieValueOnServer({ name: TACHYON_USER_COOKIE_NAME, res });
}

/**
 * Returns the current user session by looking for the presence of the
 * authentication cookie.
 */
export function getCurrentUserOnServer(
  req: Request,
  res: Response,
): CurrentUser {
  // If identity stops issuing permanent tokens we'll want isLoggedIn to check
  // for the token expiration and not just the presence of the auth-token cookie
  const authorizationToken = getAuthTokenOnServer(req, res);

  return authorizationToken
    ? { authorizationToken, isLoggedIn: true }
    : { authorizationToken: null, isLoggedIn: false };
}
