import type { Request, Response } from 'express';
import { random } from 'faker';
import { TACHYON_USER_COOKIE_NAME } from 'tachyon-auth';
import {
  GLOBAL_COOKIE_NAMES,
  clearCookieValueOnServer,
  getAndExtendCookieValueOnServer,
} from 'tachyon-server-utils';
import { getCurrentUserOnServer, logoutOnServer } from '.';

const mockClearCookieValueOnServer = clearCookieValueOnServer as jest.Mock;
const mockGetCookieValueOnServer = getAndExtendCookieValueOnServer as jest.Mock;
jest.mock('tachyon-server-utils', () => ({
  ...jest.requireActual('tachyon-server-utils'),
  clearCookieValueOnServer: jest.fn(),
  getAndExtendCookieValueOnServer: jest.fn(),
}));

describe(getCurrentUserOnServer, () => {
  beforeEach(() => {
    mockGetCookieValueOnServer.mockReset();
  });

  it('returns currentUser when logged in', () => {
    const req = {} as Request;
    const res = {} as Response;
    const authToken = random.alphaNumeric(30);
    mockGetCookieValueOnServer.mockImplementationOnce(() => authToken);

    expect(getCurrentUserOnServer(req, res)).toEqual({
      authorizationToken: authToken,
      isLoggedIn: true,
    });
  });

  it('returns currentUser when not logged in', () => {
    const req = {} as Request;
    const res = {} as Response;
    mockGetCookieValueOnServer.mockImplementationOnce(() => undefined);

    expect(getCurrentUserOnServer(req, res)).toEqual({
      authorizationToken: null,
      isLoggedIn: false,
    });
  });
});

describe(logoutOnServer, () => {
  beforeEach(() => {
    mockClearCookieValueOnServer.mockReset();
  });

  it('removes the auth-token and user cookie', () => {
    const res = { cookie: jest.fn() } as any as Response;
    logoutOnServer(res);

    expect(mockClearCookieValueOnServer).toHaveBeenCalledWith({
      name: GLOBAL_COOKIE_NAMES.AUTH_TOKEN_COOKIE_NAME,
      res,
    });
    expect(mockClearCookieValueOnServer).toHaveBeenCalledWith({
      name: TACHYON_USER_COOKIE_NAME,
      res,
    });
  });
});
