import type { FC } from 'react';
import { createContext, useCallback, useMemo, useState } from 'react';
import { useConstCallback, useEffectOnce } from 'tachyon-utils-react';
import { isBrowser } from 'tachyon-utils-stdlib';
import type { CurrentUser, TachyonUser } from '../cookies';
import {
  clearAuthTokenInBrowser,
  clearTachyonUserCookieInBrowser,
  getCurrentUserInBrowser,
  getTachyonUserCookieInBrowser,
  setAuthTokenInBrowser,
  setTachyonUserCookieInBrowser,
} from '../cookies';
import type { OAuthTokenResponse } from '../types';

export type OAuthToken = Pick<
  OAuthTokenResponse,
  'access_token' | 'refresh_token'
>;

export type CurrentUserContext = {
  getAuthToken: () => CurrentUser['authorizationToken'];
  getUserTracking: () => TachyonUser | undefined;
  loggedIn: boolean;
  login: (tokenResponse: OAuthToken, user?: TachyonUser) => void;
  logout: () => void;
};

// istanbul ignore next: trivial
export const currentUserContext = createContext<CurrentUserContext>({
  getAuthToken: () => '',
  getUserTracking: () => undefined,
  loggedIn: false,
  login: () => undefined,
  logout: () => undefined,
});

export type CurrentUserRootProps = {
  /**
   * Set's the login status for use on the server. This state will be automatically recalculated client side.
   */
  isLoggedInServerside?: boolean;
  /**
   * Invoked when the user successfully logs in.
   */
  onLogin?: (token: OAuthToken) => void;
  /**
   * Invoked when the user is logged out.
   */
  onLogout?: () => void;
};

/**
 * Mount in your application root when using `useCurrentUser`
 * and/or `useDeviceCodeFlowAuth`.
 */
export const CurrentUserRoot: FC<CurrentUserRootProps> = ({
  children,
  isLoggedInServerside = false,
  onLogin,
  onLogout,
}) => {
  const [loggedIn, setLoggedIn] = useState(isLoggedInServerside);

  useEffectOnce(() => {
    setLoggedIn(getCurrentUserInBrowser().isLoggedIn);
  });

  const login = useCallback(
    (token, user) => {
      setAuthTokenInBrowser(token);
      if (user) {
        setTachyonUserCookieInBrowser(user);
      }
      setLoggedIn(true);
      onLogin?.(token);
    },
    [onLogin],
  );

  const logout = useCallback(() => {
    clearAuthTokenInBrowser();
    clearTachyonUserCookieInBrowser();
    setLoggedIn(false);
    onLogout?.();
  }, [onLogout]);

  // Return an empty string on the server, consumers should not be consuming
  // the auth token in server client code at risk of it being serialized and included
  // in HTML creating a security risk
  const getAuthToken = useConstCallback(() =>
    isBrowser() ? getCurrentUserInBrowser().authorizationToken : '',
  );

  // istanbul ignore next: trivial
  // tracking data is only reported client side
  const getUserTracking = useConstCallback(() =>
    isBrowser() ? getTachyonUserCookieInBrowser() : undefined,
  );

  const context = useMemo(
    () => ({
      getAuthToken,
      getUserTracking,
      loggedIn,
      login,
      logout,
    }),
    [getAuthToken, getUserTracking, loggedIn, login, logout],
  );

  return <currentUserContext.Provider children={children} value={context} />;
};

CurrentUserRoot.displayName = 'CurrentUserRoot';
