import type { FC } from 'react';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import type { TachyonUser } from '../cookies';
import {
  clearAuthTokenInBrowser,
  clearTachyonUserCookieInBrowser,
  getCurrentUserInBrowser,
  setAuthTokenInBrowser,
  setTachyonUserCookieInBrowser,
} from '../cookies';
import type { OAuthTokenResponse } from '../types';
import type { CurrentUserContext } from '.';
import { CurrentUserRoot, currentUserContext } from '.';

jest.mock('../cookies', () => ({
  clearAuthTokenInBrowser: jest.fn(),
  clearTachyonUserCookieInBrowser: jest.fn(),
  getCurrentUserInBrowser: jest.fn(() => ({
    authorizationToken: undefined,
    isLoggedIn: false,
  })),
  setAuthTokenInBrowser: jest.fn(),
  setTachyonUserCookieInBrowser: jest.fn(),
}));
const mockGetCurrentUserInBrowser = getCurrentUserInBrowser as jest.Mock;
const mockSetTachyonUserCookieInBrowser =
  setTachyonUserCookieInBrowser as jest.Mock;

const CtxReader: FC<CurrentUserContext> = () => null;

describe(CurrentUserRoot, () => {
  const setup = createMountWrapperFactory(CurrentUserRoot, () => ({
    children: (
      <currentUserContext.Consumer children={(ctx) => <CtxReader {...ctx} />} />
    ),
    isLoggedInServerside: false,
    onLogin: jest.fn(),
    onLogout: jest.fn(),
  }));

  it('sets isLoggedInServerside client side', () => {
    mockGetCurrentUserInBrowser.mockImplementationOnce(() => ({
      isLoggedIn: true,
    }));
    const { wrapper } = setup({ isLoggedInServerside: false });
    expect(wrapper.find(CtxReader)).toHaveProp({ loggedIn: true });
  });

  it('logs in', () => {
    const { props, wrapper } = setup({ isLoggedInServerside: false });
    const mockToken: OAuthTokenResponse = {
      access_token: 'foo',
      device_metadata: 'qr_code',
      expires_in: 1,
      refresh_token: 'bar',
      token_type: 'baz',
    };

    expect(props.onLogin).not.toHaveBeenCalled();
    expect(setAuthTokenInBrowser).not.toHaveBeenCalled();

    wrapper.find(CtxReader).invoke('login')(mockToken);

    expect(props.onLogin).toHaveBeenCalledTimes(1);
    expect(props.onLogin).toHaveBeenCalledWith(mockToken);
    expect(setAuthTokenInBrowser).toHaveBeenCalledTimes(1);
    expect(setAuthTokenInBrowser).toHaveBeenCalledWith(mockToken);

    expect(wrapper.find(CtxReader)).toHaveProp({ loggedIn: true });
  });

  it('sets the TachyonUser cookie if user is provided', () => {
    const { wrapper } = setup();
    const mockToken: OAuthTokenResponse = {
      access_token: 'foo',
      device_metadata: 'qr_code',
      expires_in: 1,
      refresh_token: 'bar',
      token_type: 'baz',
    };

    const mockUser: TachyonUser = {
      id: '1',
      login: 'foo',
      version: '1',
    };

    wrapper.find(CtxReader).invoke('login')(mockToken, mockUser);
    expect(mockSetTachyonUserCookieInBrowser).toHaveBeenCalledWith(mockUser);
  });

  it('logs out', () => {
    const { props, wrapper } = setup({ isLoggedInServerside: true });

    expect(props.onLogout).not.toHaveBeenCalled();
    expect(clearAuthTokenInBrowser).not.toHaveBeenCalled();
    expect(clearTachyonUserCookieInBrowser).not.toHaveBeenCalled();

    wrapper.find(CtxReader).invoke('logout')();

    expect(props.onLogout).toHaveBeenCalledTimes(1);
    expect(wrapper.find(CtxReader)).toHaveProp({ loggedIn: false });
    expect(clearAuthTokenInBrowser).toHaveBeenCalledTimes(1);
    expect(clearTachyonUserCookieInBrowser).toHaveBeenCalledTimes(1);
  });

  it('returns auth token', () => {
    const mockAuthToken = 'foo';

    mockGetCurrentUserInBrowser
      .mockImplementationOnce(() => ({
        authorizationToken: mockAuthToken,
        isLoggedIn: true,
      }))
      .mockImplementationOnce(() => ({
        authorizationToken: mockAuthToken,
        isLoggedIn: true,
      }));

    const { wrapper } = setup();
    expect(wrapper.find(CtxReader).prop('getAuthToken')()).toEqual(
      mockAuthToken,
    );
  });
});
