import { random } from 'faker';
import {
  GLOBAL_COOKIE_NAMES,
  clearCookieValue,
  getAndExtendCookieValue,
  setCookieValue,
} from 'tachyon-utils-twitch';
import {
  clearAuthTokenInBrowser,
  getCurrentUserInBrowser,
  setAuthTokenInBrowser,
} from '.';

const mockClearCookieValue = clearCookieValue as jest.Mock;
const mockGetCookieValue = getAndExtendCookieValue as jest.Mock;
const mockSetCookieValue = setCookieValue as jest.Mock;
jest.mock('tachyon-utils-twitch', () => ({
  ...jest.requireActual('tachyon-utils-twitch'),
  clearCookieValue: jest.fn(),
  getAndExtendCookieValue: jest.fn(),
  setCookieValue: jest.fn(),
}));

describe(getCurrentUserInBrowser, () => {
  beforeEach(() => {
    mockGetCookieValue.mockReset();
  });

  it('returns currentUser when logged in', () => {
    const authToken = random.alphaNumeric(30);
    mockGetCookieValue.mockImplementationOnce(() => authToken);

    expect(getCurrentUserInBrowser()).toEqual({
      authorizationToken: authToken,
      isLoggedIn: true,
    });
  });

  it('returns currentUser when not logged in', () => {
    mockGetCookieValue.mockImplementationOnce(() => undefined);

    expect(getCurrentUserInBrowser()).toEqual({
      authorizationToken: null,
      isLoggedIn: false,
    });
  });
});

describe(setAuthTokenInBrowser, () => {
  beforeEach(() => {
    mockSetCookieValue.mockReset();
  });

  it('sets the auth-token cookie', () => {
    const token = { access_token: random.alphaNumeric(30) };
    setAuthTokenInBrowser(token);

    expect(mockSetCookieValue).toHaveBeenCalledWith({
      name: GLOBAL_COOKIE_NAMES.AUTH_TOKEN_COOKIE_NAME,
      value: token.access_token,
    });
  });
});

describe(clearAuthTokenInBrowser, () => {
  beforeEach(() => {
    mockClearCookieValue.mockReset();
  });

  it('removes the auth-token cookie', () => {
    clearAuthTokenInBrowser();

    expect(mockClearCookieValue).toHaveBeenCalledWith({
      name: GLOBAL_COOKIE_NAMES.AUTH_TOKEN_COOKIE_NAME,
    });
  });
});
