import { useEffect, useRef, useState } from 'react';
import { useDeviceCodeFlowTracking } from 'tachyon-event-tracker';
import {
  useForceUpdate,
  useMountedState,
  useUnmount,
} from 'tachyon-utils-react';
import type { PollingFn } from 'tachyon-utils-stdlib';
import { CANCELLED_PROMISE_ERROR_MESSAGE, poll } from 'tachyon-utils-stdlib';
import type { LoginMedium } from '../../interpol';
import {
  getDeviceCode,
  getDeviceCodeInfo,
  getToken,
  getTokenInfo,
} from '../../interpol';
import type { DeviceCodeInfo, OAuthTokenResponse } from '../../types';
import { useCurrentUser } from '../useCurrentUser';

type DeviceCodeFlowState =
  | {
      status: 'DEVICE_CODE_FAIL';
      value: Error;
    }
  | {
      status: 'DEVICE_CODE_LOADING';
      value: undefined;
    }
  | {
      status: 'LOGGED_IN';
      value: undefined;
    }
  | {
      status: 'LOGIN_POLLING';
      value: DeviceCodeInfo;
    }
  | {
      status: 'LOGIN_SUCCESS';
      value: LoginMedium;
    };

/*
 * Polls the Interpol service until a successful login is made using the
 * user_code associated with the device code or the device code expiration is
 * reached.
 *
 * Upon successful login, sets the auth-token cookie and updates the logged in
 * status for the user.
 *
 * Note: Expects that the `CurrentUserRoot` context is mounted in the component
 * tree.
 *
 * In a SSR context we only hit the 'LOGGED_IN' or 'DEVICE_CODE_LOADING' state.
 * Code fetching and token polling only occur client side.
 *
 * State machine:
 *
 *                          +---------------------+
 *                          |      LOGGED_IN      |
 *                          +---------------------+
 * +------------------+     +---------------------+
 * | DEVICE_CODE_FAIL | <-- | DEVICE_CODE_LOADING | <+
 * +------------------+     +---------------------+  |
 *                            |                      |
 *                            |                      | device code expiration
 *                            v                      |
 *                          +---------------------+  |
 *                          |    LOGIN_POLLING    | -+
 *                          +---------------------+
 *                            |
 *                            |
 *                            v
 *                          +---------------------+
 *                          |    LOGIN_SUCCESS    |
 *                          +---------------------+
 */
export function useDeviceCodeFlowAuth(clientId: string): DeviceCodeFlowState {
  const { loggedIn, login } = useCurrentUser();
  const track = useDeviceCodeFlowTracking();
  const isMounted = useMountedState();
  const [tokenRequestCount, requestNewDeviceCode] = useForceUpdate();
  const [DCFState, setDCFState] = useState<DeviceCodeFlowState>(
    loggedIn
      ? { status: 'LOGGED_IN', value: undefined }
      : { status: 'DEVICE_CODE_LOADING', value: undefined },
  );

  const pollFn = useRef<PollingFn<() => Promise<OAuthTokenResponse>>>();
  useUnmount(() => {
    pollFn.current?.cancel();
  });

  useEffect(() => {
    // Handles the case of an already logged in user. This wouldn't otherwise be
    // an error case, but there isn't any reason for the user to go through the
    // login flow.
    if (loggedIn || !isMounted()) {
      return;
    }

    getDeviceCode(clientId)
      .then((code) => {
        if (!isMounted()) {
          return;
        }

        setDCFState({
          status: 'LOGIN_POLLING',
          value: getDeviceCodeInfo(code),
        });

        pollFn.current = poll(() => getToken(clientId, code.device_code), {
          intervalMs: code.interval * 1000,
          timeoutMs: code.expires_in * 1000,
        });

        pollFn
          .current()
          .then((token) => {
            if (isMounted()) {
              login(token);
              track({
                context:
                  token.device_metadata === 'qr_code' ? 'qr_code' : 'manual',
                flow_state: 'activated',
              });
              setDCFState({
                status: 'LOGIN_SUCCESS',
                value: getTokenInfo(token),
              });
            }
          })
          .catch((error: Error) => {
            if (
              isMounted() &&
              error.message !== CANCELLED_PROMISE_ERROR_MESSAGE
            ) {
              setDCFState({ status: 'DEVICE_CODE_LOADING', value: undefined });
              requestNewDeviceCode();
            }
          });
      })
      .catch((error: Error) => {
        if (isMounted()) {
          setDCFState({ status: 'DEVICE_CODE_FAIL', value: error });
        }
      });
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [tokenRequestCount]);

  return DCFState;
}
