import { renderHook } from '@testing-library/react-hooks';
import { poll } from 'tachyon-utils-stdlib';
import { getDeviceCode, getToken } from '../../interpol';
import { useCurrentUser } from '../useCurrentUser';
import { useDeviceCodeFlowAuth } from '.';

const mockCancel = jest.fn();
jest.mock('tachyon-utils-stdlib', () => ({
  ...jest.requireActual('tachyon-utils-stdlib'),
  poll: jest.fn((fn: any) => {
    fn.cancel = mockCancel;
    return fn;
  }),
}));

jest.mock('../../interpol', () => ({
  ...jest.requireActual('../../interpol'),
  getDeviceCode: jest.fn(),
  getToken: jest.fn(),
}));

jest.mock('../useCurrentUser', () => ({
  useCurrentUser: jest.fn(() => ({
    loggedIn: false,
    login: jest.fn(),
  })),
}));

const mockGetDeviceCode = getDeviceCode as jest.Mock;
const mockGetToken = getToken as jest.Mock;
const mockPoll = poll as jest.Mock;
const mockUseCurrentUser = useCurrentUser as jest.Mock;

describe(useDeviceCodeFlowAuth, () => {
  const clientId = 'foo';
  const mockCode = {
    device_code: 'GmRhmhcxhwAzkoEqiMEg_DnyEysNkuNhszIySk9eS',
    expires_in: 1800,
    interval: 5,
    user_code: 'WDJBMJHT',
    verification_uri: 'https://www.twitch.tv/activate',
  };

  it('LOGGED_IN status when user is already logged in', () => {
    mockUseCurrentUser.mockImplementationOnce(() => ({ loggedIn: true }));

    const { result } = renderHook(useDeviceCodeFlowAuth, {
      initialProps: clientId,
    });

    expect(mockGetDeviceCode).toHaveBeenCalledTimes(0);

    expect(result.current).toEqual({ status: 'LOGGED_IN', value: undefined });
  });

  it('DEVICE_CODE_LOADING status when user is not logged in', () => {
    const mockPromise = {
      then: jest.fn(() => ({ catch: jest.fn() })),
    };
    mockGetDeviceCode.mockImplementationOnce(() => mockPromise);

    const { result } = renderHook(useDeviceCodeFlowAuth, {
      initialProps: clientId,
    });

    expect(mockGetDeviceCode).toHaveBeenCalledTimes(1);

    expect(result.current).toEqual({
      status: 'DEVICE_CODE_LOADING',
      value: undefined,
    });
  });

  it('DEVICE_CODE_FAIL status when device code fetch fails', async () => {
    const error = new Error('Oh Noes');
    mockGetDeviceCode.mockImplementationOnce(() => Promise.reject(error));

    const { result, waitForNextUpdate } = renderHook(useDeviceCodeFlowAuth, {
      initialProps: clientId,
    });

    await waitForNextUpdate();

    expect(result.current).toEqual({
      status: 'DEVICE_CODE_FAIL',
      value: error,
    });
  });

  it('LOGIN_POLLING status when device code fetch succeeds', async () => {
    mockGetDeviceCode.mockImplementationOnce(() => Promise.resolve(mockCode));
    const mockPromise = {
      then: jest.fn(() => ({ catch: jest.fn() })),
    };
    mockGetToken.mockImplementationOnce(() => mockPromise);

    const { result, waitForNextUpdate } = renderHook(useDeviceCodeFlowAuth, {
      initialProps: clientId,
    });

    await waitForNextUpdate();

    expect(mockPoll).toHaveBeenCalledWith(expect.any(Function), {
      intervalMs: mockCode.interval * 1000,
      timeoutMs: mockCode.expires_in * 1000,
    });

    expect(result.current).toEqual({
      status: 'LOGIN_POLLING',
      value: {
        formattedUserCode: 'WDJB-MJHT',
        qrCodeUri:
          'https://www.twitch.tv/activate?device-code=WDJBMJHT&device-metadata=qr_code',
        userCode: 'WDJBMJHT',
        verificationUri: 'https://www.twitch.tv/activate',
      },
    });
  });

  it('LOGIN_SUCCESS status when token request succeeds', async () => {
    const mockLogin = jest.fn();
    mockUseCurrentUser.mockImplementationOnce(() => ({
      loggedIn: false,
      login: mockLogin,
    }));
    mockGetDeviceCode.mockImplementationOnce(() => Promise.resolve(mockCode));
    mockGetToken.mockImplementationOnce(() =>
      Promise.resolve({ access_token: 'foo' }),
    );

    const { result, waitForNextUpdate } = renderHook(useDeviceCodeFlowAuth, {
      initialProps: clientId,
    });

    await waitForNextUpdate();

    expect(mockLogin).toHaveBeenCalledWith({ access_token: 'foo' });

    expect(result.current).toEqual({
      status: 'LOGIN_SUCCESS',
      value: 'manual',
    });
  });

  it('refetches the device code when token request expires', async () => {
    const mockPromise = {
      then: jest.fn(() => ({ catch: jest.fn() })),
    };
    mockGetDeviceCode
      .mockImplementationOnce(() => Promise.resolve(mockCode))
      .mockImplementationOnce(() => mockPromise);
    mockGetToken.mockImplementationOnce(() =>
      Promise.reject(new Error('timeout!')),
    );

    const { result, waitForNextUpdate } = renderHook(useDeviceCodeFlowAuth, {
      initialProps: clientId,
    });

    await waitForNextUpdate();

    expect(mockGetDeviceCode).toHaveBeenCalledTimes(2);

    expect(result.current).toEqual({
      status: 'DEVICE_CODE_LOADING',
      value: undefined,
    });
  });
});
