import { stringify } from 'query-string';
import { HTTPStatusCode } from 'tachyon-type-library';
import { formattedUserCode, getQrCodeUri } from '../formatters';
import type {
  DeviceCodeInfo,
  OAuthDeviceCodeResponse,
  OAuthTokenResponse,
} from '../types';

/*
 * Currently we're requesting a subset of scopes outlined here:
 * https://git.xarth.tv/twitch-apps/universal_viewer/pull/2877/files#diff-5ca8fd113dbe99398ba5eeeb61f11ca1R64
 *
 * TODO: Audit scopes: https://jira.twitch.com/browse/EMP-2789
 */
const OAUTH_SCOPES = [
  'channel_read',
  'chat:read',
  'user_blocks_edit',
  'user_blocks_read',
  'user_follows_edit',
  'user_read',
].join(' ');

export async function getDeviceCode(
  clientId: string,
): Promise<OAuthDeviceCodeResponse> {
  const response = await fetch('https://id.twitch.tv/oauth2/device', {
    body: stringify({
      client_id: clientId,
      scopes: OAUTH_SCOPES,
    }),
    headers: {
      Accept: 'application/json',
      'Content-Type': 'application/x-www-form-urlencoded',
    },
    method: 'POST',
  });

  const body = await response.json();
  if (response.status !== HTTPStatusCode.OK) {
    throw body;
  }
  return body;
}

export async function getToken(
  clientId: string,
  deviceCode: string,
): Promise<OAuthTokenResponse> {
  const response = await fetch('https://id.twitch.tv/oauth2/token', {
    body: stringify({
      client_id: clientId,
      device_code: deviceCode,
      grant_type: 'urn:ietf:params:oauth:grant-type:device_code',
    }),
    headers: {
      Accept: 'application/json',
      'Content-Type': 'application/x-www-form-urlencoded',
    },
    method: 'POST',
  });

  const body = await response.json();
  if (response.status !== HTTPStatusCode.OK) {
    throw body;
  }
  return body;
}

export function getDeviceCodeInfo(
  deviceCode: OAuthDeviceCodeResponse,
): DeviceCodeInfo {
  return {
    formattedUserCode: formattedUserCode(deviceCode),
    qrCodeUri: getQrCodeUri(deviceCode),
    userCode: deviceCode.user_code,
    verificationUri: deviceCode.verification_uri,
  };
}

// The method the user logged in with
export type LoginMedium = 'manual' | 'qr_code';

export function getTokenInfo(token: OAuthTokenResponse): LoginMedium {
  return token.device_metadata === 'qr_code' ? 'qr_code' : 'manual';
}
