import type { ReactChild } from 'react';
import { Component } from 'react';
import type {
  ChatEvent,
  MessageEvent,
  MessagePart,
  ResubscriptionEvent,
  SubscriptionEvent,
} from 'tachyon-chat';
import type { IntlProps } from 'tachyon-intl';
import type { Padding, PaddingValues } from 'twitch-core-ui';
import {
  Background,
  Color,
  CoreLink,
  CoreText,
  Layout,
  TextType,
  WordBreak,
} from 'twitch-core-ui';
import type { BadgeListProps } from './BadgeList';
import { BadgeList, DEFAULT_BADGE_SIZE } from './BadgeList';
import { ChatAction } from './ChatAction';
import type { ChatEmoteProps } from './ChatEmote';
import { ChatEmote, DEFAULT_EMOTE_SIZE } from './ChatEmote';
import { ChatModeration } from './ChatModeration';
import { ChatPost } from './ChatPost';
import { Username } from './Username';

export interface ChatMessageProps extends Pick<IntlProps, 'intl'> {
  className?: string;
  message: ChatEvent;
  sizes?:
    | Partial<
        Pick<BadgeListProps, 'badgeSize'> & Pick<ChatEmoteProps, 'emoteSize'>
      >
    | undefined;
}

export class ChatMessage extends Component<ChatMessageProps> {
  public static displayName = 'ChatMessage';

  public override shouldComponentUpdate(nextProps: ChatMessageProps): boolean {
    if (nextProps.message.id !== this.props.message.id) {
      return true;
    }

    return (
      (nextProps.message as MessageEvent).deleted !==
      (this.props.message as MessageEvent).deleted
    );
  }

  public override render(): JSX.Element {
    const padding = ['SUBSCRIPTION', 'RESUBSCRIPTION'].includes(
      this.props.message.type,
    )
      ? undefined
      : ({ x: 2, y: 0.5 } as PaddingValues);

    return (
      <Layout
        as="li"
        // TODO CORE-UI-TYPES: Update core-ui types to accept undefined
        // https://jira.xarth.tv/browse/EMP-4748
        className={this.props.className as string}
        // TODO CORE-UI-TYPES: Update core-ui types to accept undefined
        // https://jira.xarth.tv/browse/EMP-4748
        padding={padding as Padding}
      >
        {this.renderMessageContent()}
      </Layout>
    );
  }

  public renderMessageContent(): JSX.Element | undefined {
    const {
      intl: { formatMessage },
      message,
    } = this.props;

    switch (message.type) {
      case 'CONNECTED':
        return (
          <CoreText color={Color.Alt2} type={TextType.Span}>
            {formatMessage('Welcome to the chat room!', 'ChatMessage')}
          </CoreText>
        );
      case 'DISCONNECTED':
        if (message.reason) {
          return (
            <CoreText color={Color.Alt2} type={TextType.Span}>
              {formatMessage(
                'You have been disconnected from chat for the following reason: {reason}',
                {
                  reason: message.reason,
                },
                'ChatMessage',
              )}
            </CoreText>
          );
        }

        return (
          <CoreText color={Color.Alt2} type={TextType.Span}>
            {formatMessage(
              'You have been disconnected from chat.',
              'ChatMessage',
            )}
          </CoreText>
        );
      case 'RECONNECT':
        return (
          <CoreText color={Color.Alt2} type={TextType.Span}>
            {formatMessage(
              'Sorry, we were unable to connect to chat. Attempting to reconnect...',
              'ChatMessage',
            )}
          </CoreText>
        );
      case 'HOSTING':
        return (
          <CoreText color={Color.Alt2} type={TextType.Span}>
            {formatMessage(
              'Now hosting {channel}.',
              {
                channel: message.hostedChannel,
              },
              'ChatMessage',
            )}
          </CoreText>
        );
      case 'UNHOST':
        return (
          <CoreText color={Color.Alt2} type={TextType.Span}>
            {formatMessage('No longer hosting.', 'ChatMessage')}
          </CoreText>
        );
      case 'POST':
        return (
          message.messageParts && (
            <ChatPost
              badgeSize={this.badgeSize}
              message={message}
              parts={this.messageParts(message.messageParts, message.deleted)}
            />
          )
        );
      case 'ACTION':
        return (
          message.messageParts && (
            <ChatAction
              badgeSize={this.badgeSize}
              message={message}
              parts={this.messageParts(message.messageParts, message.deleted)}
            />
          )
        );
      case 'BAN':
      case 'TIMEOUT':
        return <ChatModeration message={message} />;
      case 'SUBSCRIPTION':
      case 'RESUBSCRIPTION':
        return (
          <Layout
            background={Background.Alt2}
            borderLeft
            fullHeight
            fullWidth
            padding={{ x: 2, y: 0.5 }}
          >
            <CoreText color={Color.Alt} type={TextType.Span}>
              {message.type === 'SUBSCRIPTION'
                ? this.subscriptionMessage(message)
                : this.resubscriptionMessage(message)}
            </CoreText>
          </Layout>
        );
      default:
        return undefined;
    }
  }

  private deletedMessage(): JSX.Element {
    const { formatMessage } = this.props.intl;

    return (
      <CoreText color={Color.Link} key="deleted" type={TextType.Span}>
        {`<${formatMessage('message deleted', 'ChatMessage')}>`}
      </CoreText>
    );
  }

  private messageParts(
    parts: MessagePart[],
    isDeleted: boolean | undefined,
  ): JSX.Element | JSX.Element[] {
    if (isDeleted) {
      return this.deletedMessage();
    }

    return parts.reduce<JSX.Element[]>((msgParts, part, idx) => {
      let newPart: JSX.Element | JSX.Element[];
      switch (part.type) {
        case 'TEXT':
          newPart = <span key={idx}>{part.content}</span>;
          break;
        case 'LINK':
          newPart = (
            <CoreText key={idx} wordBreak={WordBreak.BreakWord}>
              <CoreLink
                linkTo="/deferToRenderLink"
                renderLink={({ children }) => (
                  <a children={children} href={part.content.url} />
                )}
                targetBlank
              >
                {part.content.displayText}
              </CoreLink>
            </CoreText>
          );
          break;
        case 'EMOTE':
          newPart = (
            <ChatEmote emoteSize={this.emoteSize} key={idx} {...part.content} />
          );
          break;
        default:
          return part as never;
      }

      return msgParts.concat(newPart);
    }, []);
  }

  private subscriptionMessage(message: SubscriptionEvent): ReactChild {
    if (message.isPrime) {
      return this.formatPrimeSubscribeMessage(
        message.channel,
        message.username,
      );
    }

    return this.justSubscribeMessage(message.username);
  }

  private justSubscribeMessage(username: string): string {
    return this.props.intl.formatMessage(
      '{username} just subscribed!',
      { username },
      'ChatMessage',
    );
  }

  private resubscriptionMessage(
    message: ResubscriptionEvent,
  ): Array<JSX.Element | string> {
    const resubscriptionMessageComponents: Array<JSX.Element | string> = [];
    const { formatMessage } = this.props.intl;

    // this is mostly to satisfy TS, not sure how user could be undefined for a resub
    if (message.user) {
      if (message.isPrime) {
        resubscriptionMessageComponents.push(
          <span key="resub-prime">
            {this.formatPrimeSubscribeMessage(
              message.channel,
              message.user.usernameDisplay,
            )}
          </span>,
          <span key="resub-separator"> </span>,
        );
      }

      // this currently doesn't work with tmi.js
      if (message.months) {
        resubscriptionMessageComponents.push(
          <span key="resub-months">
            {formatMessage(
              '{username} subscribed for {months} months in a row!',
              {
                months: message.months,
                username: message.user.usernameDisplay,
              },
              'ChatMessage',
            )}
          </span>,
        );
      }

      // this is a fallback since months is broken
      if (!message.isPrime && !message.months) {
        resubscriptionMessageComponents.push(
          <span key="resub">
            {this.justSubscribeMessage(message.user.usernameDisplay)}
          </span>,
        );
      }
    }

    if (message.messageParts && message.badges && message.user) {
      resubscriptionMessageComponents.push(
        <div key="resub-message">
          <BadgeList badgeSize={this.badgeSize} badges={message.badges} />
          <Username user={message.user} />
          <CoreText color={Color.Base} type={TextType.Span}>
            <span key="separator">{' : '}</span>
            {this.messageParts(message.messageParts, message.deleted)}
          </CoreText>
        </div>,
      );
    }

    return resubscriptionMessageComponents;
  }

  private formatPrimeSubscribeMessage(
    channel: string,
    usernameDisplay: string,
  ): ReactChild {
    const { formatMessage } = this.props.intl;

    // TODO: this should be done via x-tags in the format message below
    // https://jira.twitch.com/browse/EMP-2770
    const service = (
      <CoreLink
        linkTo="/deferToRenderLink"
        renderLink={() => (
          <a
            href={`https://twitch.amazon.com/prime?ref=subscriptionMessage&channel=${channel}`}
          />
        )}
        targetBlank
      >
        Twitch Prime
      </CoreLink>
    );

    return formatMessage(
      '{username} just subscribed with {service}!',
      {
        service,
        username: usernameDisplay,
      },
      'ChatMessage',
    );
  }

  private get badgeSize(): string {
    return this.props.sizes?.badgeSize || DEFAULT_BADGE_SIZE;
  }

  private get emoteSize(): string {
    return this.props.sizes?.emoteSize || DEFAULT_EMOTE_SIZE;
  }
}
