import {
  buildTestConnectionStatusMessage,
  buildTestHostingStatusMessage,
  buildTestMessageEvent,
  buildTestMessageEventWithCheer,
  buildTestModerationEvent,
  buildTestResubscriptionEvent,
  buildTestSubscriptionEvent,
} from 'tachyon-chat';
import { DefaultTwitchIntl } from 'tachyon-intl';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { ChatAction } from './ChatAction';
import { ChatEmote } from './ChatEmote';
import { ChatModeration } from './ChatModeration';
import { ChatPost } from './ChatPost';
import { ChatMessage } from '.';

describe(ChatMessage, () => {
  const setup = createMountWrapperFactory(ChatMessage, () => ({
    intl: DefaultTwitchIntl,
    message: {
      id: 'ABC123',
      type: 'CONNECTED',
    },
  }));

  describe('StatusEvent', () => {
    it('renders a CONNECTED type message', () => {
      const { wrapper } = setup({
        message: buildTestConnectionStatusMessage(),
      });
      expect(wrapper).toHaveText('Welcome to the chat room!');
    });

    it('renders a RECONNECT type message', () => {
      const { wrapper } = setup({
        message: buildTestConnectionStatusMessage('RECONNECT'),
      });
      expect(wrapper).toHaveText(
        'Sorry, we were unable to connect to chat. Attempting to reconnect...',
      );
    });

    it('renders a DISCONNECTED type message', () => {
      const { wrapper } = setup({
        message: buildTestConnectionStatusMessage('DISCONNECTED'),
      });
      expect(wrapper).toHaveText('You have been disconnected from chat.');
    });

    it('renders a DISCONNECTED with reason type message', () => {
      const { wrapper } = setup({
        message: buildTestConnectionStatusMessage('DISCONNECTED', true),
      });
      expect(wrapper).toHaveText(
        'You have been disconnected from chat for the following reason: lost in a series of tubes',
      );
    });

    it('renders a HOSTING with reason type message', () => {
      const { wrapper } = setup({
        message: buildTestHostingStatusMessage(),
      });
      expect(wrapper).toHaveText('Now hosting monstercat.');
    });

    it('renders a UNHOST with reason type message', () => {
      const { wrapper } = setup({
        message: buildTestHostingStatusMessage('UNHOST'),
      });
      expect(wrapper).toHaveText('No longer hosting.');
    });
  });

  describe('MessageEvent', () => {
    it('renders a POST type message', () => {
      const { props, wrapper } = setup({
        message: buildTestMessageEvent(),
      });
      expect(wrapper.find(ChatPost)).toHaveProp({ message: props.message });
    });

    it('renders a POST type message with cheer', () => {
      const cheerMessage = buildTestMessageEventWithCheer();
      const { props, wrapper } = setup({
        message: cheerMessage,
      });
      expect(wrapper.find(ChatPost)).toHaveProp({ message: props.message });
      expect(wrapper.find(ChatEmote)).toHaveProp(
        cheerMessage.messageParts?.[0].content as any,
      );
    });

    it('renders a deleted POST type message', () => {
      const { wrapper } = setup({
        message: buildTestMessageEvent(undefined, undefined, true),
      });
      expect(wrapper).toHaveText('Voxel : <message deleted>');
    });

    it('renders an ACTION type message', () => {
      const { props, wrapper } = setup({
        message: buildTestMessageEvent(undefined, 'ACTION'),
      });
      expect(wrapper.find(ChatAction)).toHaveProp({ message: props.message });
    });

    it('renders a deleted ACTION type message', () => {
      const { wrapper } = setup({
        message: buildTestMessageEvent(undefined, 'ACTION', true),
      });
      expect(wrapper).toHaveText('Voxel <message deleted>');
    });
  });

  it('renders a moderation message', () => {
    const { props, wrapper } = setup({
      message: buildTestModerationEvent(),
    });
    expect(wrapper.find(ChatModeration)).toHaveProp({ message: props.message });
  });

  describe('SubscribeEvent', () => {
    it('renders a SUBSCRIPTION without prime event', () => {
      const { wrapper } = setup({
        message: buildTestSubscriptionEvent(),
      });
      expect(wrapper).toHaveText('voxel just subscribed!');
    });

    it('renders a SUBSCRIPTION with prime event', () => {
      const { wrapper } = setup({
        message: buildTestSubscriptionEvent(true),
      });
      expect(wrapper).toHaveText('voxel just subscribed with [object Object]!');
    });

    it('renders a RESUBSCRIPTION without prime and without message event', () => {
      const { wrapper } = setup({
        message: buildTestResubscriptionEvent(),
      });
      expect(wrapper).toHaveText('Voxel subscribed for 3 months in a row!');
    });

    it('renders a RESUBSCRIPTION with prime and without message event', () => {
      const { wrapper } = setup({
        message: buildTestResubscriptionEvent(true),
      });
      expect(wrapper).toHaveText(
        'Voxel just subscribed with [object Object]! Voxel subscribed for 3 months in a row!',
      );
    });

    it('renders a RESUBSCRIPTION without prime and with message event', () => {
      const { wrapper } = setup({
        message: buildTestResubscriptionEvent(false, true),
      });
      expect(wrapper).toHaveText(
        'Voxel subscribed for 3 months in a row!Voxel :  text ',
      );
    });

    it('renders a RESUBSCRIPTION with prime and with message event', () => {
      const { wrapper } = setup({
        message: buildTestResubscriptionEvent(true, true),
      });
      expect(wrapper).toHaveText(
        'Voxel just subscribed with [object Object]! Voxel subscribed for 3 months in a row!Voxel :  text ',
      );
    });

    it('renders a RESUBSCRIPTION with deleted message event', () => {
      const { wrapper } = setup({
        message: buildTestResubscriptionEvent(undefined, true, true),
      });
      expect(wrapper).toHaveText(
        'Voxel subscribed for 3 months in a row!Voxel : <message deleted>',
      );
    });
  });
});
