import type { FC } from 'react';
import { useEffect, useMemo } from 'react';
import type { ChatWorkerBridge, ChatWorkerInstance } from 'tachyon-chat';
import { createChatWorkerBridge } from 'tachyon-chat';
import { useConst, useUnmount } from 'tachyon-utils-react';
import type { ChatContext } from '../context';
import { chatContext } from '../context';
import { useChatRootState } from './useChatRootState';

/**
 * Props for the Tachyon Chat Root.
 */
export type ChatRootProps = {
  /**
   * The number of messages to keep in buffer to be rendered.
   */
  bufferSize: number;
  channel: {
    id: string;
    login: string;
  };
  clientApiId: string;
  getWorker: () => ChatWorkerInstance;
};

export const ChatRoot: FC<ChatRootProps> = ({
  bufferSize,
  channel: { id, login },
  children,
  clientApiId,
  getWorker,
}) => {
  const [state, dispatch] = useChatRootState();

  const bridge = useConst<ChatWorkerBridge | undefined>(() => {
    if (typeof window !== 'undefined') {
      return createChatWorkerBridge({
        bufferSize,
        clientApiId,
        getWorker,
        updateChatMessages: (messages) => {
          dispatch({ messages, type: 'updateMessages' });
        },
      });
    }
  });

  useEffect(() => {
    if (!id || !login) {
      dispatch({ type: 'reset' });
    }
    bridge?.({ id, login });
  }, [id, login, bridge, dispatch]);

  useUnmount(() => {
    // force worker cleanup
    bridge?.({});
  });

  const ctx = useMemo<ChatContext>(
    () => ({
      chatMessages: state.chatMessages,
      pause: () => dispatch({ type: 'pause' }),
      unpause: () => dispatch({ type: 'unpause' }),
    }),
    [state, dispatch],
  );

  return <chatContext.Provider children={children} value={ctx} />;
};

ChatRoot.displayName = 'ChatRoot';
