import { datatype, lorem } from 'faker';
import type { FC } from 'react';
import { act } from 'react-dom/test-utils';
import { buildTestMessageEvent, createChatWorkerBridge } from 'tachyon-chat';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import type { ChatContext } from '../context';
import { useChat } from '../context';
import type { ChatRootState } from './useChatRootState';
import { initialChatRootState } from './useChatRootState';
import { ChatRoot } from '.';

const mockBridge = jest.fn();
jest.mock('tachyon-chat', () => ({
  ...jest.requireActual('tachyon-chat'),
  createChatWorkerBridge: jest.fn(() => mockBridge),
}));
const mockCreateChatWorkerBridge = createChatWorkerBridge as jest.Mock;

let mockState: ChatRootState = initialChatRootState;
const mockDispatch = jest.fn();
jest.mock('./useChatRootState', () => ({
  ...jest.requireActual('./useChatRootState'),
  useChatRootState: jest.fn(() => [mockState, mockDispatch]),
}));

describe(ChatRoot, () => {
  beforeEach(() => {
    mockState = initialChatRootState;
  });

  const setup = createMountWrapperFactory(ChatRoot, () => ({
    bufferSize: datatype.number(),
    channel: { id: datatype.uuid(), login: lorem.word() },
    clientApiId: lorem.word(),
    getWorker: jest.fn(),
  }));

  const messages: ChatContext['chatMessages'] = [
    buildTestMessageEvent(),
    buildTestMessageEvent(),
  ];

  it('creates a chat worker bridge with proper functionality', () => {
    const { props } = setup();
    expect(mockCreateChatWorkerBridge).toHaveBeenLastCalledWith({
      bufferSize: props.bufferSize,
      clientApiId: props.clientApiId,
      getWorker: props.getWorker,
      updateChatMessages: expect.any(Function),
    });

    const updateChatMessages =
      mockCreateChatWorkerBridge.mock.calls[0][0].updateChatMessages;
    expect(mockDispatch).not.toHaveBeenCalled();

    act(() => {
      updateChatMessages(messages);
    });
    expect(mockDispatch).toHaveBeenLastCalledWith({
      messages,
      type: 'updateMessages',
    });
  });

  it('sends down functional context', () => {
    mockState.chatMessages = messages;
    const Ctx: FC = () => {
      const ctx = useChat();
      return <div data-ctx={ctx} />;
    };

    const { wrapper } = setup({ children: <Ctx /> });
    const ctx: ChatContext = wrapper.find('div').prop('data-ctx');

    expect(ctx.chatMessages).toEqual(messages);
    expect(mockDispatch).not.toHaveBeenCalled();

    ctx.pause();
    expect(mockDispatch).toHaveBeenCalledTimes(1);
    expect(mockDispatch).toHaveBeenLastCalledWith({ type: 'pause' });

    ctx.unpause();
    expect(mockDispatch).toHaveBeenCalledTimes(2);
    expect(mockDispatch).toHaveBeenLastCalledWith({ type: 'unpause' });
  });

  describe('communciating channel to bridge', () => {
    it('passes the initial channel to bridge', () => {
      expect(mockBridge).not.toHaveBeenCalled();
      const { props } = setup();

      expect(mockBridge).toHaveBeenCalledTimes(1);
      expect(mockBridge).toHaveBeenLastCalledWith(props.channel);
      expect(mockDispatch).not.toHaveBeenCalled();
    });

    it('passes channel updates to the bridge', () => {
      const { props, wrapper } = setup();

      expect(mockBridge).toHaveBeenCalledTimes(1);
      expect(mockBridge).toHaveBeenLastCalledWith(props.channel);

      const newChannel = {
        id: datatype.uuid(),
        login: lorem.word(),
      };
      wrapper.setProps({ channel: newChannel });
      expect(mockBridge).toHaveBeenCalledTimes(2);
      expect(mockBridge).toHaveBeenLastCalledWith(newChannel);
      expect(mockDispatch).not.toHaveBeenCalled();
    });

    it('defensively resets state for missing channel id', () => {
      setup({ channel: { id: undefined } as any });
      expect(mockBridge).toHaveBeenCalledTimes(1);
      expect(mockDispatch).toHaveBeenCalledTimes(1);
      expect(mockDispatch).toHaveBeenLastCalledWith({ type: 'reset' });
    });

    it('defensively resets state for missing channel login', () => {
      setup({ channel: { login: undefined } as any });
      expect(mockBridge).toHaveBeenCalledTimes(1);
      expect(mockDispatch).toHaveBeenCalledTimes(1);
      expect(mockDispatch).toHaveBeenLastCalledWith({ type: 'reset' });
    });

    it('passes empty object to bridge on unmount', () => {
      const { props, wrapper } = setup();

      expect(mockBridge).toHaveBeenCalledTimes(1);
      expect(mockBridge).toHaveBeenLastCalledWith(props.channel);

      act(() => {
        wrapper.unmount();
      });
      expect(mockBridge).toHaveBeenCalledTimes(2);
      expect(mockBridge).toHaveBeenLastCalledWith({});
    });
  });
});
