import type { ChatEvent } from '../../events';

export type ChatEventMessageHandler = (message: { data: ChatEvent }) => void;

/**
 * Maintains a buffer of chat message history (defaults to 250) for a given channel including
 * pruning of messages that have been moderated.
 */
export class ChatBuffer {
  private buffer: ChatEvent[];
  private moderatedUsers: Set<string>;
  private _hasNewMessages: boolean;
  private maxSize: number;

  constructor(maxSize: number) {
    this.buffer = [];
    this.moderatedUsers = new Set();
    this._hasNewMessages = false;
    this.maxSize = maxSize;
  }

  public consumeChatEvent = (message: { data: ChatEvent }): void => {
    const data = message.data;
    switch (data.type) {
      case 'BAN':
      case 'TIMEOUT':
        const { duration, username } = data;
        if (this.moderatedUsers.has(username)) {
          return;
        }
        this.buffer.forEach((event) => {
          if (event.type === 'POST' || event.type === 'ACTION') {
            if (username === event.user.username && !event.deleted) {
              event.deleted = true;
            }
          } else if (event.type === 'RESUBSCRIPTION' && event.messageParts) {
            if (
              event.user &&
              event.user.username === username &&
              !event.deleted
            ) {
              event.deleted = true;
            }
          }
        });
        this.moderatedUsers.add(username);
        if (duration) {
          // this is for debouncing duplicate moderation messages during the timeout
          setTimeout(this.unmoderateUser(username), duration * 1000);
        }
        break;
      case 'DISCONNECTED':
        // if we want to show these, we'd need the same de-duping as reconnect so just delete the next line
        return;
      case 'RECONNECT':
        // tmi.js sends duplicate connection-related events frequently
        if (this.buffer.find((e) => e.id === data.id)) {
          return;
        }
    }
    // We used to do this in `getCurrentMessages`, but by moving this here, we
    // wind up with a lot of smaller array thrashes instead of big one each time
    // the UI is going to update; we also thrash an existing array instead of
    // creating an entire new one each time. This is better because it allows us
    // to avoid the messages of "Warning: setTimeout took longer than x ms"
    // which are meant to indicate that you are causing a pileup by having your
    // timeout take longer than its interval timing. This is also slightly more
    // efficient when moderation actions come in because there will be fewer
    // messages to search through to update their deleted status. These benefits
    // outweigh the costs of incremental de-allocations (partly since the GC
    // will let them pile up and clear them all out at once during an idle
    // moment).
    this.buffer.push(data);
    if (this.buffer.length > this.maxSize) {
      this.buffer.shift();
    }
    this._hasNewMessages = true;
  };

  public getCurrentMessages(): ReadonlyArray<ChatEvent> {
    this._hasNewMessages = false;

    return this.buffer;
  }

  get hasNewMessages(): boolean {
    return this._hasNewMessages;
  }

  // length() and moderatedUserCount() are only for testability
  get length(): number {
    return this.buffer.length;
  }

  get moderatedUserCount(): number {
    return this.moderatedUsers.size;
  }

  private unmoderateUser = (username: string): (() => void) => {
    return (): void => {
      this.moderatedUsers.delete(username);
    };
  };
}
