import type { MessageEvent } from '../../events';
import {
  buildTestConnectionStatusMessage,
  buildTestHostingStatusMessage,
  buildTestMessageEvent,
  buildTestModerationEvent,
  buildTestResubscriptionEvent,
  buildTestSubscriptionEvent,
} from '../../test-utils';
import { ChatBuffer } from '.';

describe(ChatBuffer, () => {
  let chatBuffer: ChatBuffer;
  beforeEach(() => {
    chatBuffer = new ChatBuffer(3);
  });

  it('constructs with empty buffer and no moderated users', () => {
    expect(chatBuffer).toHaveLength(0);
    expect(chatBuffer.moderatedUserCount).toEqual(0);
  });

  describe('buffer management', () => {
    it('adds a single chat message to the buffer and updates hasNewMessages state', () => {
      const event = buildTestMessageEvent();
      expect(chatBuffer.hasNewMessages).toEqual(false);

      chatBuffer.consumeChatEvent({ data: event });
      expect(chatBuffer).toHaveLength(1);
      expect(chatBuffer.moderatedUserCount).toEqual(0);
      expect(chatBuffer.hasNewMessages).toEqual(true);

      const bufferArray = chatBuffer.getCurrentMessages();
      expect(bufferArray[0]).toBe(event);
      expect(chatBuffer.hasNewMessages).toEqual(false);
    });

    it('adds messages to fill the buffer without rolling over', () => {
      const events = [
        buildTestMessageEvent(),
        buildTestMessageEvent(),
        buildTestMessageEvent(),
      ];
      events.forEach((event) => chatBuffer.consumeChatEvent({ data: event }));

      expect(chatBuffer).toHaveLength(3);
      expect(chatBuffer.moderatedUserCount).toEqual(0);

      const bufferArray = chatBuffer.getCurrentMessages();
      expect(bufferArray[0]).toBe(events[0]);
      expect(bufferArray[1]).toBe(events[1]);
      expect(bufferArray[2]).toBe(events[2]);
    });

    it('adds messages beyond max size while truncating and preserving order', () => {
      const events = [
        buildTestMessageEvent(),
        buildTestMessageEvent(),
        buildTestMessageEvent(),
        buildTestMessageEvent(),
      ];
      events.forEach((event) => chatBuffer.consumeChatEvent({ data: event }));

      expect(chatBuffer).toHaveLength(3);
      expect(chatBuffer.moderatedUserCount).toEqual(0);

      const bufferArray = chatBuffer.getCurrentMessages();
      expect(bufferArray[0]).toBe(events[1]);
      expect(bufferArray[1]).toBe(events[2]);
      expect(bufferArray[2]).toBe(events[3]);
      expect(bufferArray).toHaveLength(3);
    });

    it('handles multiple rounds of overflow and getCurrentMessages() calls', () => {
      expect(chatBuffer.hasNewMessages).toEqual(false);

      chatBuffer.consumeChatEvent({ data: buildTestMessageEvent() });

      expect(chatBuffer).toHaveLength(1);
      expect(chatBuffer.hasNewMessages).toEqual(true);

      chatBuffer.getCurrentMessages();
      expect(chatBuffer.hasNewMessages).toEqual(false);

      chatBuffer.consumeChatEvent({ data: buildTestMessageEvent() });
      chatBuffer.consumeChatEvent({ data: buildTestMessageEvent() });
      chatBuffer.consumeChatEvent({ data: buildTestMessageEvent() });

      expect(chatBuffer).toHaveLength(3);
      expect(chatBuffer.hasNewMessages).toEqual(true);

      chatBuffer.getCurrentMessages();
      expect(chatBuffer.hasNewMessages).toEqual(false);

      chatBuffer.consumeChatEvent({ data: buildTestMessageEvent() });
      chatBuffer.consumeChatEvent({ data: buildTestMessageEvent() });
      chatBuffer.consumeChatEvent({ data: buildTestMessageEvent() });
      chatBuffer.consumeChatEvent({ data: buildTestMessageEvent() });
      chatBuffer.consumeChatEvent({ data: buildTestMessageEvent() });

      expect(chatBuffer).toHaveLength(3);
      expect(chatBuffer.hasNewMessages).toEqual(true);

      chatBuffer.getCurrentMessages();
      expect(chatBuffer.hasNewMessages).toEqual(false);
    });
  });

  describe('moderation', () => {
    it('moderates a single message', () => {
      const chatEvent = buildTestMessageEvent();
      chatBuffer.consumeChatEvent({ data: chatEvent });
      expect(chatEvent.deleted).toEqual(false);

      chatBuffer.consumeChatEvent({ data: buildTestModerationEvent() });

      expect(chatBuffer).toHaveLength(2);
      expect(chatEvent.deleted).toEqual(true);
      expect(chatBuffer.moderatedUserCount).toEqual(1);
    });

    it('moderates multiple messages and truncates as necessary', () => {
      const events = [
        buildTestMessageEvent(),
        buildTestMessageEvent(),
        buildTestMessageEvent(),
      ];
      events.forEach((event) => chatBuffer.consumeChatEvent({ data: event }));

      const modEvent = buildTestModerationEvent();
      chatBuffer.consumeChatEvent({ data: modEvent });
      events.forEach((event) => expect(event.deleted).toEqual(true));

      const bufferArray = chatBuffer.getCurrentMessages();
      expect(bufferArray[0]).toBe(events[1]);
      expect((bufferArray[0] as MessageEvent).deleted).toBe(true);
      expect(bufferArray[1]).toBe(events[2]);
      expect((bufferArray[1] as MessageEvent).deleted).toBe(true);
      expect(bufferArray[2]).toBe(modEvent);
      expect(chatBuffer.moderatedUserCount).toEqual(1);
    });

    it('does not stack duplicate moderation messages', () => {
      const chatEvent = buildTestMessageEvent();
      chatBuffer.consumeChatEvent({ data: chatEvent });
      expect(chatEvent.deleted).toEqual(false);

      const modEvent = buildTestModerationEvent();
      chatBuffer.consumeChatEvent({ data: modEvent });
      chatBuffer.consumeChatEvent({ data: buildTestModerationEvent() });

      expect(chatBuffer).toHaveLength(2);
      expect(chatEvent.deleted).toEqual(true);

      const bufferArray = chatBuffer.getCurrentMessages();
      expect(bufferArray[0]).toBe(chatEvent);
      expect(bufferArray[1]).toBe(modEvent);
      expect(chatBuffer.moderatedUserCount).toEqual(1);
    });

    it('only moderates messages for the moderated user', () => {
      const chatEventUser1 = buildTestMessageEvent();
      const chatEventUser2 = buildTestMessageEvent('anotherUser');

      chatBuffer.consumeChatEvent({ data: chatEventUser1 });
      chatBuffer.consumeChatEvent({ data: chatEventUser2 });

      chatBuffer.consumeChatEvent({ data: buildTestModerationEvent() });

      expect(chatEventUser1.deleted).toEqual(true);
      expect(chatEventUser2.deleted).toEqual(false);
      expect(chatBuffer.moderatedUserCount).toEqual(1);
    });

    it('does not error when first message received is moderation', () => {
      const modEvent = buildTestModerationEvent();
      chatBuffer.consumeChatEvent({ data: modEvent });

      expect(chatBuffer).toHaveLength(1);
      expect(chatBuffer.moderatedUserCount).toEqual(1);
    });

    it('clears the moderated user list after duration', () => {
      chatBuffer.consumeChatEvent({ data: buildTestModerationEvent() });
      expect(chatBuffer.moderatedUserCount).toEqual(1);

      jest.advanceTimersByTime(1);
      expect(chatBuffer.moderatedUserCount).toEqual(0);
    });
  });

  describe('consuming message types', () => {
    it('adds a single subscription message to the buffer and updates hasNewMessages state', () => {
      const event = buildTestSubscriptionEvent();
      expect(chatBuffer.hasNewMessages).toEqual(false);

      chatBuffer.consumeChatEvent({ data: event });
      expect(chatBuffer).toHaveLength(1);
      expect(chatBuffer.moderatedUserCount).toEqual(0);
      expect(chatBuffer.hasNewMessages).toEqual(true);

      expect(chatBuffer.getCurrentMessages()[0]).toBe(event);
    });

    it('adds a single resubscription message to the buffer and updates hasNewMessages state', () => {
      const event = buildTestResubscriptionEvent();
      expect(chatBuffer.hasNewMessages).toEqual(false);

      chatBuffer.consumeChatEvent({ data: event });
      expect(chatBuffer).toHaveLength(1);
      expect(chatBuffer.moderatedUserCount).toEqual(0);
      expect(chatBuffer.hasNewMessages).toEqual(true);

      expect(chatBuffer.getCurrentMessages()[0]).toBe(event);
    });

    it('moderates a single resubscription message', () => {
      const subEvent = buildTestResubscriptionEvent(undefined, true);
      chatBuffer.consumeChatEvent({ data: subEvent });
      expect(subEvent.deleted).toEqual(false);

      chatBuffer.consumeChatEvent({ data: buildTestModerationEvent() });

      expect(chatBuffer).toHaveLength(2);
      expect(subEvent.deleted).toEqual(true);
      expect(chatBuffer.moderatedUserCount).toEqual(1);
    });

    it('adds connected status events to the buffer and updates hasNewMessages state', () => {
      const event = buildTestConnectionStatusMessage();
      expect(chatBuffer.hasNewMessages).toEqual(false);

      chatBuffer.consumeChatEvent({ data: event });
      expect(chatBuffer).toHaveLength(1);
      expect(chatBuffer.moderatedUserCount).toEqual(0);
      expect(chatBuffer.hasNewMessages).toEqual(true);

      expect(chatBuffer.getCurrentMessages()[0]).toBe(event);
    });

    it('adds reconnection status events to the buffer and updates hasNewMessages state', () => {
      const event = buildTestConnectionStatusMessage('RECONNECT');
      expect(chatBuffer.hasNewMessages).toEqual(false);

      chatBuffer.consumeChatEvent({ data: event });
      expect(chatBuffer).toHaveLength(1);
      expect(chatBuffer.moderatedUserCount).toEqual(0);
      expect(chatBuffer.hasNewMessages).toEqual(true);

      expect(chatBuffer.getCurrentMessages()[0]).toBe(event);
    });

    it('does not add duplicate reconnection status events to the buffer', () => {
      chatBuffer.consumeChatEvent({
        data: buildTestConnectionStatusMessage('RECONNECT'),
      });
      chatBuffer.consumeChatEvent({
        data: buildTestConnectionStatusMessage('RECONNECT'),
      });
      expect(chatBuffer).toHaveLength(1);
    });

    it('does not add disconnected events to the buffer', () => {
      chatBuffer.consumeChatEvent({
        data: buildTestConnectionStatusMessage('DISCONNECTED'),
      });
      expect(chatBuffer).toHaveLength(0);
    });

    it('adds hosting status events to the buffer and updates hasNewMessages state', () => {
      const event = buildTestHostingStatusMessage();
      expect(chatBuffer.hasNewMessages).toEqual(false);

      chatBuffer.consumeChatEvent({ data: event });
      expect(chatBuffer).toHaveLength(1);
      expect(chatBuffer.moderatedUserCount).toEqual(0);
      expect(chatBuffer.hasNewMessages).toEqual(true);

      expect(chatBuffer.getCurrentMessages()[0]).toBe(event);
    });

    it('adds unhost status events to the buffer and updates hasNewMessages state', () => {
      const event = buildTestHostingStatusMessage('UNHOST');
      expect(chatBuffer.hasNewMessages).toEqual(false);

      chatBuffer.consumeChatEvent({ data: event });
      expect(chatBuffer).toHaveLength(1);
      expect(chatBuffer.moderatedUserCount).toEqual(0);
      expect(chatBuffer.hasNewMessages).toEqual(true);

      expect(chatBuffer.getCurrentMessages()[0]).toBe(event);
    });
  });
});
