import { POLYFILL_SCRIPT_ID } from 'tachyon-polyfill';
import type { ChatEvent } from '../../events';
import type { WorkerMessage } from '../../types';
import type { ChatEventMessageHandler } from '../ChatBuffer';
import { ChatBuffer } from '../ChatBuffer';

// This represents 5Hz, which is somewhat arbitrary but was chosen to allow for
// highly frequent updates so that the UI is not jerky but without wasting tons
// of CPU cycles. Previously it was 60Hz due to use of requestAnimationFrame and
// that was way too much processing for no visible benefit.
const CHAT_POLL_TIMEOUT_MS = 200;

export type ChatWorkerInstance = {
  addEventListener: (
    messageType: 'message',
    messageHandler: ChatEventMessageHandler,
  ) => void;
  postMessage: (message: WorkerMessage) => void;
  removeEventListener: (
    messageType: 'message',
    messageHandler: ChatEventMessageHandler,
  ) => void;
};

type ChatConfig = {
  bufferSize: number;
  clientApiId: string;
  getWorker: () => ChatWorkerInstance;
  updateChatMessages: (chatMessages: ReadonlyArray<ChatEvent>) => void;
};

type ChatResources = {
  buffer: ChatBuffer;
  timeoutHandle: number;
  type: 'ACTIVE';
  worker: ChatWorkerInstance;
};

type NullChatResources = {
  buffer: null;
  timeoutHandle: null;
  type: 'NULL';
  worker: null;
};

type ChatTargetDetails = {
  id?: string;
  login?: string;
};

export type ChatWorkerBridge = (chatTargetDetails: ChatTargetDetails) => void;

/**
 * Initializes and manages interactions with Chat's Web Worker.
 */
export function createChatWorkerBridge({
  bufferSize,
  clientApiId,
  getWorker,
  updateChatMessages,
}: ChatConfig): ChatWorkerBridge {
  function initializeChatResources(): ChatResources {
    return {
      buffer: new ChatBuffer(bufferSize),
      timeoutHandle: window.setTimeout(updateChatState, CHAT_POLL_TIMEOUT_MS),
      type: 'ACTIVE',
      worker: getWorker(),
    };
  }

  function initializeNullChatResources(): NullChatResources {
    return {
      buffer: null,
      timeoutHandle: null,
      type: 'NULL',
      worker: null,
    };
  }

  let channelLogin: string | undefined;
  let channelID: string | undefined;
  let chatResources: ChatResources | NullChatResources =
    initializeNullChatResources();

  function updateChatState(): void {
    if (!chatResources.buffer) {
      // we're cleaning up and exiting
      return;
    }

    if (chatResources.buffer.hasNewMessages) {
      updateChatMessages(chatResources.buffer.getCurrentMessages());
    }

    chatResources.timeoutHandle = window.setTimeout(
      updateChatState,
      CHAT_POLL_TIMEOUT_MS,
    );
  }

  return ({ id, login }) => {
    // if either id or channel are missing, kill worker if exists
    if (!login || !id) {
      if (chatResources.type === 'ACTIVE') {
        chatResources.worker.postMessage({
          command: 'CHAT_WORKER_DISCONNECTED',
        });

        chatResources.worker.removeEventListener(
          'message',
          chatResources.buffer.consumeChatEvent,
        );

        clearTimeout(chatResources.timeoutHandle);
        chatResources = initializeNullChatResources();
      }
    } else if (chatResources.type === 'NULL') {
      chatResources = initializeChatResources();
      // else if there isn't a chat worker, build one
      const polyfillElement = document.getElementById(POLYFILL_SCRIPT_ID);
      const polyfillURI = polyfillElement?.getAttribute('src') ?? null;

      chatResources.worker.postMessage({
        command: 'CHAT_WORKER_CONNECTED',
        payload: {
          channelID: id,
          channelLogin: login,
          clientApiId,
          polyfillURI,
        },
      });

      chatResources.worker.addEventListener(
        'message',
        chatResources.buffer.consumeChatEvent,
      );

      chatResources.timeoutHandle = window.setTimeout(
        updateChatState,
        CHAT_POLL_TIMEOUT_MS,
      );
    } else if (login !== channelLogin || id !== channelID) {
      // else if the target chat channel has changed, change channel
      clearTimeout(chatResources.timeoutHandle);

      chatResources.worker.removeEventListener(
        'message',
        chatResources.buffer.consumeChatEvent,
      );
      chatResources.buffer = new ChatBuffer(bufferSize);
      chatResources.worker.postMessage({
        command: 'CHAT_WORKER_CHANNEL_CHANGED',
        payload: {
          channelID: id,
          channelLogin: login,
        },
      });

      chatResources.worker.addEventListener(
        'message',
        chatResources.buffer.consumeChatEvent,
      );
      chatResources.timeoutHandle = window.setTimeout(
        updateChatState,
        CHAT_POLL_TIMEOUT_MS,
      );
    }

    channelLogin = login;
    channelID = id;
  };
}
