import { titleCase } from 'tachyon-utils-stdlib';
import type {
  ChatEvent,
  ConnectionStatusEvent,
  ConnectionStatusEventType,
  HostingStatusEvent,
  HostingStatusEventType,
  MessageEvent,
  MessageEventType,
  ModerationEvent,
  ModerationEventType,
  ResubscriptionEvent,
  SubscriptionEvent,
} from './events';
import type { UserStatePayload } from './types';
import type { MessagePart } from './utils';
import { BITS_COLOR_MAP, idCreator, makeCheermoteURL } from './utils';

// istanbul ignore file: test utils

export const TEST_MESSAGE = 'Kappa text';

export const TEST_USERSTATE: UserStatePayload = {
  badges: { warcraft: 'horde' },
  bits: undefined,
  color: '#aabbcc',
  'display-name': 'Voxel',
  emotes: { 25: ['0-4'] },
  id: '1234567890',
  'user-id': '12345',
  username: 'voxel',
};

const TEXT_MESSAGE_PART: MessagePart = {
  content: ' text ',
  type: 'TEXT',
};

const EMOTE_MESSAGE_PART: MessagePart = {
  content: {
    alt: 'Kappa',
    images: {
      '1x': 'https://static-cdn.jtvnw.net/emoticons/v1/25/1.0',
      '2x': 'https://static-cdn.jtvnw.net/emoticons/v1/25/2.0',
      '4x': 'https://static-cdn.jtvnw.net/emoticons/v1/25/3.0',
    },
  },
  type: 'EMOTE',
};

const LINK_MESSAGE_PART: MessagePart = {
  content: {
    displayText: 'chat.rocks',
    url: 'https://chat.rocks',
  },
  type: 'LINK',
};

let idCounter = 0;
export function buildTestMessageEvent(
  username = 'voxel',
  type: MessageEventType = 'POST',
  deleted = false,
): MessageEvent {
  return {
    badges: [
      {
        alt: 'For the Horde!',
        images: {
          '1x': 'https://static-cdn.jtvnw.net/badges/v1/de8b26b6-fd28-4e6c-bc89-3d597343800d/1',
          '2x': 'https://static-cdn.jtvnw.net/badges/v1/de8b26b6-fd28-4e6c-bc89-3d597343800d/2',
          '4x': 'https://static-cdn.jtvnw.net/badges/v1/de8b26b6-fd28-4e6c-bc89-3d597343800d/3',
        },
      },
    ],
    deleted,
    id: idCreator(`${username}-msg-${idCounter++}`),
    messageParts: [EMOTE_MESSAGE_PART, TEXT_MESSAGE_PART, LINK_MESSAGE_PART],
    type,
    user: {
      color: '#aabbcc',
      username,
      usernameDisplay: titleCase(username),
    },
  };
}

export function buildTestMessageEventWithCheer(
  username = 'voxel',
): MessageEvent {
  return {
    badges: [],
    deleted: false,
    id: idCreator(`${username}-cheer-${idCounter++}`),
    messageParts: [
      {
        content: {
          alt: 'Cheer',
          cheerAmount: 100,
          cheerColor: BITS_COLOR_MAP.purple,
          images: {
            '1.5x': makeCheermoteURL('cheer', 100, 1.5),
            '1x': makeCheermoteURL('cheer', 100, 1),
            '2x': makeCheermoteURL('cheer', 100, 2),
            '3x': makeCheermoteURL('cheer', 100, 3),
            '4x': makeCheermoteURL('cheer', 100, 4),
          },
        },
        type: 'EMOTE',
      },
    ],
    type: 'POST',
    user: {
      color: '#aabbcc',
      username,
      usernameDisplay: titleCase(username),
    },
  };
}

export function buildTestModerationEvent(
  username = 'voxel',
  type: ModerationEventType = 'TIMEOUT',
  reason: string | null = 'for being too awesome',
): ModerationEvent {
  return {
    duration: 0.001,
    id: idCreator(`${username}-mod-${idCounter++}`),
    reason,
    type,
    username,
  };
}

export function buildTestSubscriptionEvent(isPrime = false): SubscriptionEvent {
  return {
    channel: 'lirik',
    id: idCreator(`voxel-sub-${idCounter++}`),
    isPrime,
    type: 'SUBSCRIPTION',
    username: 'voxel',
  };
}

export function buildTestResubscriptionEvent(
  isPrime = false,
  withMessage = false,
  deleted = false,
): ResubscriptionEvent {
  return {
    badges: [],
    channel: 'lirik',
    deleted,
    id: idCreator(`voxel-resub-${idCounter++}`),
    isPrime,
    messageParts: withMessage ? [TEXT_MESSAGE_PART] : undefined,
    months: 3,
    type: 'RESUBSCRIPTION',
    user: {
      color: 'Toasted Meringue',
      username: 'voxel',
      usernameDisplay: 'Voxel',
    },
  };
}

export function buildTestConnectionStatusMessage(
  type: ConnectionStatusEventType = 'CONNECTED',
  withReason = false,
): ConnectionStatusEvent {
  if (type === 'DISCONNECTED') {
    return {
      id: 'yolo',
      reason: withReason ? 'lost in a series of tubes' : undefined,
      type: 'DISCONNECTED',
    };
  }

  return {
    id: 'yolo',
    type,
  } as ConnectionStatusEvent;
}

export function buildTestHostingStatusMessage(
  type: HostingStatusEventType = 'HOSTING',
): HostingStatusEvent {
  if (type === 'HOSTING') {
    return {
      channel: 'voxel',
      hostedChannel: 'monstercat',
      id: 'hostapalooza',
      type: 'HOSTING',
    };
  }

  return {
    channel: 'voxel',
    id: 'no-more-hosta',
    type: 'UNHOST',
  };
}

export function createTestEvents(): ChatEvent[] {
  const count = 100;

  return Array(count)
    .fill(0)
    .map((_value, index) => {
      if (index >= 98) {
        return buildTestResubscriptionEvent(!!(index % 2), !!(index % 2));
      } else if (index >= 95) {
        return buildTestSubscriptionEvent(!!(index % 2));
      } else if (index >= 91) {
        return buildTestModerationEvent();
      } else if (index >= 90) {
        return buildTestMessageEventWithCheer();
      }
      return buildTestMessageEvent();
    });
}
