import { titleCase } from 'tachyon-utils-stdlib';
import type { Badge, UserStatePayload, UserStatePayloadEmotes } from '../types';

// via https://dev.twitch.tv/docs/v5/reference/bits/
const BitsRegExp =
  /\b(cheer|doodlecheer|pogchamp|showlove|pride|heyguys|frankerz|seemsgood|party|kappa|dansgame|elegiggle|trihard|kreygasm|4head|swiftrage|notlikethis|failfish|vohiyo|pjsalt|mrdestructoid|bday|ripcheer|shamrock|bitboss|streamlabs|muxy|holidaycheer|goal|charity|anon)(\d+)\b/gi;

/**
 * Link parsing rules
 * - no immediately adjacent illegal chars on front or back, but they must be captured and separated by a space
 * - protocol is optional
 * - any number of subdomains
 * - tld must be between 2 and 6 characters
 */
const LinkRegExp =
  /([^\w@#%\-+=:~])?(?:(https?:\/\/)?(?:[\w@#%\-+=:~]+\.)+[a-z]{2,6}(?:\/[\w./@#%&()\-+=:?~]*)?)([^\w./@#%&()\-+=:?~]|\s|$)/g;

export type EmoteData = {
  alt?: string;
  cheerAmount?: number;
  cheerColor?: string;
  images: {
    [scale: string]: string;
  };
};

type EmotePosition = {
  data: EmoteData;
  endIndex: number;
  startIndex: number;
};

type TextMessagePart = {
  content: string;
  type: 'TEXT';
};

export type LinkData = {
  displayText: string;
  url: string;
};

type LinkMessagePart = {
  content: LinkData;
  type: 'LINK';
};

type EmoteMessagePart = {
  content: EmoteData;
  type: 'EMOTE';
};

export type MessagePart = EmoteMessagePart | LinkMessagePart | TextMessagePart;

export type MessageEventUserData = {
  color: string | null;
  username: string;
  usernameDisplay: string;
};

export type MessageData = {
  badges: Badge[];
  deleted: boolean;
  messageParts: MessagePart[] | undefined;
  user: MessageEventUserData;
};

export function createMessageData(
  message: string,
  userstate: UserStatePayload,
  badges: Badge[],
): MessageData {
  return {
    badges,
    deleted: false,
    messageParts: message
      ? messagePartBuilder(message, userstate.emotes, userstate.bits)
      : undefined,
    user: {
      color: userstate.color,
      username: userstate.username,
      usernameDisplay: userstate['display-name'],
    },
  };
}

function getSymbolsInString(message: string): string[] {
  // From the article JavaScript Has a Unicode Issue https://mathiasbynens.be/notes/javascript-unicode#iterating-over-symbols
  const regexCodePoint =
    /[^\uD800-\uDFFF]|[\uD800-\uDBFF][\uDC00-\uDFFF]|[\uD800-\uDFFF]/g;

  return message.match(regexCodePoint) || [];
}

export function messagePartBuilder(
  message: string,
  emotes: UserStatePayloadEmotes | null,
  bits: number | undefined,
): MessagePart[] {
  let parts: MessagePart[] = [];
  let lastEndIndex = 0;
  const allEmotes: EmotePosition[] = Array(message.length);
  const characters = getSymbolsInString(message);
  const messageHasBits = !!bits;

  if (emotes) {
    Object.keys(emotes).forEach((emote: string) => {
      const ranges = emotes[emote];
      ranges.forEach((range: string) => {
        const [start, end] = range.split('-');
        const startIndex = parseInt(start, 10);
        const endIndex = parseInt(end, 10) + 1;
        allEmotes[startIndex] = {
          data: {
            images: {
              '1x': makeEmoteURL(emote, 1),
              '2x': makeEmoteURL(emote, 2),
              '4x': makeEmoteURL(emote, 4),
            },
          },
          endIndex,
          startIndex,
        };
      });
    });
  }

  allEmotes.forEach((emote: EmotePosition) => {
    const precedingText = characters
      .slice(lastEndIndex, emote.startIndex)
      .join('');
    if (precedingText) {
      parts = parts.concat(
        messageHasBits
          ? getCheerAndLinksOutOfText(precedingText)
          : getLinksOutOfText(precedingText),
      );
    }
    parts.push({
      content: {
        alt: characters.slice(emote.startIndex, emote.endIndex).join(''),
        ...emote.data,
      },
      type: 'EMOTE',
    });
    lastEndIndex = emote.endIndex;
  });

  const trailingText = characters.slice(lastEndIndex).join('');
  if (trailingText) {
    parts = parts.concat(
      messageHasBits
        ? getCheerAndLinksOutOfText(trailingText)
        : getLinksOutOfText(trailingText),
    );
  }

  return parts;
}

function getCheerAndLinksOutOfText(text: string): MessagePart[] {
  return getCheerOutOfText(text).reduce<MessagePart[]>(
    (textParts, part) =>
      textParts.concat(
        part.type === 'TEXT' ? getLinksOutOfText(part.content) : part,
      ),
    [],
  );
}

function getCheerOutOfText(message: string): MessagePart[] {
  const parts: MessagePart[] = [];
  let match = BitsRegExp.exec(message);
  let lastEndIndex = 0;
  while (match) {
    const startIndex = match.index;
    const emote = match[1].toLowerCase();
    const amount = parseInt(match[2], 10);
    const { color, floor } = getBitsRangeAndColor(amount);
    if (startIndex > lastEndIndex) {
      parts.push({
        content: message.slice(lastEndIndex, startIndex),
        type: 'TEXT',
      });
    }
    parts.push({
      content: {
        alt: titleCase(emote),
        cheerAmount: amount,
        cheerColor: color,
        images: {
          '1.5x': makeCheermoteURL(emote, floor, 1.5),
          '1x': makeCheermoteURL(emote, floor, 1),
          '2x': makeCheermoteURL(emote, floor, 2),
          '3x': makeCheermoteURL(emote, floor, 3),
          '4x': makeCheermoteURL(emote, floor, 4),
        },
      },
      type: 'EMOTE',
    });
    lastEndIndex = BitsRegExp.lastIndex;
    match = BitsRegExp.exec(message);
  }
  if (lastEndIndex < message.length) {
    parts.push({
      content: message.slice(lastEndIndex),
      type: 'TEXT',
    });
  }

  return parts;
}

function getLinksOutOfText(text: string): MessagePart[] {
  const parts: MessagePart[] = [];
  let lastEndIndex = 0;
  let needsTrailingSpace = false;
  let match = LinkRegExp.exec(text);
  while (match) {
    let startIndex = match.index;
    let needsPrecedingSpace = false;
    const leadingChar = match[1];
    if (leadingChar) {
      startIndex += 1;
      if (!/\s/.test(leadingChar)) {
        needsPrecedingSpace = true;
      }
    }

    if (startIndex !== lastEndIndex) {
      parts.push({
        content: `${needsTrailingSpace ? ' ' : ''}${text.slice(
          lastEndIndex,
          startIndex,
        )}${needsPrecedingSpace ? ' ' : ''}`,
        type: 'TEXT',
      });
    }
    needsTrailingSpace = false;

    let endIndex = LinkRegExp.lastIndex;
    const trailingChar = match[3];
    if (trailingChar) {
      endIndex -= 1;
      if (!/\s/.test(trailingChar)) {
        needsTrailingSpace = true;
      }
    }

    const protocol = match[2];
    parts.push({
      content: {
        displayText: text.slice(startIndex, endIndex),
        url: `${protocol ? '' : 'https://'}${text.slice(startIndex, endIndex)}`,
      },
      type: 'LINK',
    });
    lastEndIndex = endIndex;
    match = LinkRegExp.exec(text);
  }

  const trailingText = text.slice(lastEndIndex);
  if (trailingText) {
    parts.push({
      content: `${needsTrailingSpace ? ' ' : ''}${trailingText}`,
      type: 'TEXT',
    });
  }

  return parts;
}

export const BITS_COLOR_MAP = {
  blue: '#0099fe',
  gray: '#979797',
  green: '#1db2a5',
  purple: '#9c3ee8',
  red: '#f43021',
};

function getBitsRangeAndColor(amount: number): {
  color: string;
  floor: number;
} {
  let floor: number;
  let color: string;

  if (amount >= 10000) {
    floor = 10000;
    color = BITS_COLOR_MAP.red;
  } else if (amount >= 5000) {
    floor = 5000;
    color = BITS_COLOR_MAP.blue;
  } else if (amount >= 1000) {
    floor = 1000;
    color = BITS_COLOR_MAP.green;
  } else if (amount >= 100) {
    floor = 100;
    color = BITS_COLOR_MAP.purple;
  } else {
    floor = 1;
    color = BITS_COLOR_MAP.gray;
  }

  return { color, floor };
}

export function makeCheermoteURL(
  emote: string,
  clamp: number,
  size: number,
): string {
  return `https://d3aqoihi2n8ty8.cloudfront.net/actions/${emote}/light/animated/${clamp}/${size}.gif`;
}

function makeEmoteURL(emote: string, size: number): string {
  if (size === 4) {
    size = 3;
  }

  return `https://static-cdn.jtvnw.net/emoticons/v1/${emote}/${size}.0`;
}
