import type { Badge, UserStatePayloadBadges } from '../../types';

export const GLOBAL_BADGES_URL =
  'https://badges.twitch.tv/v1/badges/global/display';

export function makeChannelBadgeURL(channelID: string): string {
  return `https://badges.twitch.tv/v1/badges/channels/${channelID}/display`;
}

interface BadgePayloadVersion {
  readonly click_action: string;
  readonly click_url: string;
  readonly description: string;
  readonly image_url_1x: string;
  readonly image_url_2x: string;
  readonly image_url_4x: string;
  readonly title: string;
}

export interface MutableBadgesPayload {
  [badgeSet: string]: {
    versions: {
      [badge: string]: BadgePayloadVersion;
    };
  };
}

export interface BadgesPayload extends Readonly<MutableBadgesPayload> {}

/**
 * Retrieves global badges as well as channel badges.
 */
export class BadgerService {
  public badges: BadgesPayload;

  constructor() {
    this.badges = {};
  }

  public init(channelID: string): Promise<void> {
    return Promise.all([
      this.getGlobalBadges(),
      this.getChannelBadges(channelID),
    ]).then((badges) => {
      this.badges = this.mergeBadges(badges);
    });
  }

  public getBadgeData(badges: UserStatePayloadBadges | null): Badge[] {
    if (!badges) {
      return [];
    }

    return Object.keys(badges).reduce<Badge[]>((accum, badge) => {
      let badgeData: BadgePayloadVersion;
      try {
        badgeData = this.badges[badge].versions[badges[badge]];
        accum.push({
          alt: badgeData.description,
          images: {
            '1x': badgeData.image_url_1x,
            '2x': badgeData.image_url_2x,
            '4x': badgeData.image_url_4x,
          },
        });
      } catch {} // eslint-disable-line no-empty

      return accum;
    }, []);
  }

  public getGlobalBadges(): Promise<BadgesPayload> {
    return this.fetch(GLOBAL_BADGES_URL);
  }

  public getChannelBadges(channelID: string): Promise<BadgesPayload> {
    return this.fetch(makeChannelBadgeURL(channelID));
  }

  public fetch(url: string): Promise<BadgesPayload> {
    return fetch(url)
      .then((r) => r.json())
      .then((data) => data.badge_sets)
      .catch(() => ({}));
  }

  /**
   * Merges an array of badge payloads such that earlier badges are replaced by later badges.
   */
  private mergeBadges(payloads: MutableBadgesPayload[]): BadgesPayload {
    return payloads.reduce((accum, payload: BadgesPayload) => {
      Object.keys(payload).forEach((badgeset) => {
        if (!accum[badgeset]) {
          accum[badgeset] = { ...payload[badgeset] };
        } else {
          accum[badgeset].versions = {
            ...accum[badgeset].versions,
            ...payload[badgeset].versions,
          };
        }
      });

      return accum;
    }, {});
  }
}

export const badgerService = new BadgerService();
