import {
  GLOBAL_SUBSCRIBER_VERSION_0_URL,
  TEST_CHANNEL_ID,
  mockAllBadges,
} from './test-utils';
import { BadgerService } from '.';

describe(BadgerService, () => {
  let badger: BadgerService;

  beforeEach(() => {
    badger = new BadgerService();
  });

  describe('bare singleton', () => {
    it('constructs without badges', () => {
      expect(Object.keys(badger.badges)).toHaveLength(0);
    });
  });

  describe('with fetches', () => {
    beforeEach(() => {
      mockAllBadges();
    });

    afterEach(() => {
      fetchMock.resetMocks();
    });

    it('fetches badges on init and merges subscriber badges with channel specifics', () => {
      return badger.init(TEST_CHANNEL_ID).then(() => {
        expect(badger.badges).toMatchObject({});
        // ensure general badges are available
        expect(badger.badges.warcraft).toMatchObject({});
        // ensure general badges are still available as well
        expect(badger.badges.subscriber.versions['1']).toMatchObject({});
        // ensure channel-specific subscriber badges have overwritten general badges
        expect(badger.badges.subscriber.versions['0'].image_url_1x).not.toEqual(
          GLOBAL_SUBSCRIBER_VERSION_0_URL,
        );
      });
    });

    it('returns proper badge info from getBadgeData', () => {
      return badger.init(TEST_CHANNEL_ID).then(() => {
        const badgeData = badger.getBadgeData({
          subscriber: '12',
          warcraft: 'horde',
        });
        expect(badgeData).toHaveLength(2);
        expect(badgeData[0].alt).toEqual('1-Year Subscriber');
        expect(badgeData[1].alt).toEqual('For the Horde!');
      });
    });

    it('safely ignores and filters unknown badges', () => {
      return badger.init(TEST_CHANNEL_ID).then(() => {
        const badgeData = badger.getBadgeData({
          starcraft: 'zerg',
          subscriber: '12',
        });
        expect(badgeData).toHaveLength(1);
        expect(badgeData[0].alt).toEqual('1-Year Subscriber');
      });
    });

    it('returns empty list from getBadgeData when there are no badges', () => {
      return badger.init(TEST_CHANNEL_ID).then(() => {
        expect(badger.getBadgeData(null)).toHaveLength(0);
      });
    });
  });
});
