// eslint-disable-next-line @typescript-eslint/no-var-requires
const TMI = require('tachyon-tmi.js');

const DEBUG_TMI_CLIENT = false;

export type UserStatePayloadBadges = {
  readonly [badgeset: string]: string;
};

export type UserStatePayloadEmotes = {
  readonly [emote: string]: string[];
};

export type UserStatePayload = {
  readonly badges: UserStatePayloadBadges | null;
  readonly bits: number | undefined;
  readonly color: string | null;
  readonly 'display-name': string;
  readonly emotes: UserStatePayloadEmotes | null;
  readonly id: string;
  readonly 'user-id': string;
  readonly username: string;
};

export type SubscriptionMethods = {
  prime: boolean;
};

export type ConnectedHandler = (address: string, port: number) => void;
export type DisconnectedHandler = (reason: string) => void;
export type ReconnectHandler = () => void;

export type HostingHandler = (
  channel: string,
  target: string,
  viewers: number,
) => void;
export type UnhostHandler = (channel: string, viewers: number) => void;

export type ChatHandler = (
  channel: string,
  userstate: UserStatePayload,
  message: string,
  sentByCurrentUser: boolean,
) => void;

export type ModerationHandler = (
  channel: string,
  username: string,
  reason: string,
  duration?: number,
) => void;

export type SubscriptionHandler = (
  channel: string,
  username: string,
  methods: SubscriptionMethods,
) => void;

export type ResubscriptionHandler = (
  channel: string,
  username: string,
  months: number,
  message: string,
  userstate: UserStatePayload,
  methods: SubscriptionMethods,
) => void;

type TMIEventHandler =
  | ChatHandler
  | ConnectedHandler
  | DisconnectedHandler
  | HostingHandler
  | ModerationHandler
  | ReconnectHandler
  | ResubscriptionHandler
  | SubscriptionHandler
  | UnhostHandler;

type SocketState = 'CLOSED' | 'CLOSING' | 'CONNECTING' | 'OPEN';

export type TMIClient = {
  addListener: (event: string, handler: TMIEventHandler) => void;
  connect: () => Promise<void>;
  disconnect: () => Promise<void>;
  join: (channel: string) => Promise<void>;
  part: (channel: string) => Promise<void>;
  readyState: () => SocketState;
};

// istanbul ignore next: would mostly be testing mocks
export class ChatClient {
  public client: TMIClient;
  public currentChannel: string | undefined;

  constructor(clientApiId: string) {
    this.client = TMI.client({
      connection: {
        reconnect: true,
        secure: true,
        timeout: 2000,
      },
      options: {
        clientId: clientApiId,
        debug: DEBUG_TMI_CLIENT,
      },
    });
  }

  public connect(channel: string): Promise<void> {
    this.currentChannel = channel;
    return this.client.connect().then(() => {
      if (this.currentChannel) {
        return this.client.join(this.currentChannel);
      }
    });
  }

  public disconnect(): Promise<void> {
    return this.client.disconnect();
  }

  public changeChannel(channel: string): Promise<void> {
    const promise = this.currentChannel
      ? this.client.part(this.currentChannel)
      : Promise.resolve();

    return promise.then(() => {
      this.currentChannel = channel;
      return this.client.join(this.currentChannel);
    });
  }

  public setConnectedHandler(connectedHandler: ConnectedHandler): void {
    this.client.addListener('connected', connectedHandler);
  }

  public setDisconnectedHandler(
    disconnectedHandler: DisconnectedHandler,
  ): void {
    this.client.addListener('disconnected', disconnectedHandler);
  }

  public setReconnectHandler(reconnectHandler: ReconnectHandler): void {
    this.client.addListener('reconnect', reconnectHandler);
  }

  public setHostingHandler(hostingHandler: HostingHandler): void {
    this.client.addListener('hosting', hostingHandler);
  }

  public setUnhostHandler(unhostHandler: UnhostHandler): void {
    this.client.addListener('unhost', unhostHandler);
  }

  public setChatHandler(chatHandler: ChatHandler): void {
    this.client.addListener('chat', chatHandler);
    this.client.addListener('cheer', chatHandler);
  }

  public setActionHandler(actionHandler: ChatHandler): void {
    this.client.addListener('action', actionHandler);
  }

  public setTimeoutHandler(timeoutHandler: ModerationHandler): void {
    this.client.addListener('timeout', timeoutHandler);
  }

  public setBanHandler(banHandler: ModerationHandler): void {
    this.client.addListener('ban', banHandler);
  }

  public setSubcriptionHandler(subcriptionHandler: SubscriptionHandler): void {
    this.client.addListener('subscription', subcriptionHandler);
  }

  public setResubscriptionHandler(
    resubscriptionHandler: ResubscriptionHandler,
  ): void {
    this.client.addListener('resub', resubscriptionHandler);
  }

  public isSocketOpen(): boolean {
    return this.client.readyState() === 'OPEN';
  }
}
