import { logger } from 'tachyon-logger';
import { errorMessageFromCatch } from 'tachyon-utils-stdlib';
import type {
  ChatEvent,
  MessageEventType,
  ModerationEventType,
} from '../../events';
import {
  connectedEvent,
  disconnectedEvent,
  hostingEvent,
  messageEvent,
  moderationEvent,
  reconnectEvent,
  resubscriptionEvent,
  subscriptionEvent,
  unhostEvent,
} from '../../events';
import type { WorkerMessage } from '../../types';
import type { BadgerService } from '../BadgerService';
import { badgerService } from '../BadgerService';
import type {
  ChatHandler,
  ModerationHandler,
  SubscriptionMethods,
  UserStatePayload,
} from '../ChatClient';
import { ChatClient } from '../ChatClient';

function importScriptsWrapper(...urls: string[]): void {
  self.importScripts(...urls);
}

function postMessageWrapper(msg: any): void {
  self.postMessage(msg);
}

interface ChatServiceOpts {
  badger: BadgerService;
  importScripts: (...targets: string[]) => void;
  postMessage: (
    message: ChatEvent,
    targetOrigin?: string,
    transfer?: any[],
  ) => void;
}

/**
 * Orchestrates the Chat Web Worker.
 */
export class ChatService {
  public channelLogin: string | undefined;
  public channelID: string | undefined;
  public importScripts: (...targets: string[]) => void;
  public postMessage: (
    message: ChatEvent,
    targetOrigin?: string,
    transfer?: any[],
  ) => void;
  public badger: BadgerService;
  public client: ChatClient | null = null;

  constructor(opts: Partial<ChatServiceOpts> = {}) {
    this.importScripts = opts.importScripts || importScriptsWrapper;
    this.postMessage = opts.postMessage || postMessageWrapper;
    this.badger = opts.badger || badgerService;

    self.addEventListener('message', this.messageHandler.bind(this));
  }

  public messageHandler(message: {
    data: WorkerMessage;
  }): Promise<[void, void] | void> {
    switch (message.data.command) {
      case 'CHAT_WORKER_CONNECTED':
        const { channelID, channelLogin, clientApiId, polyfillURI } =
          message.data.payload;

        if (!polyfillURI) {
          logger.error({
            category: 'ChatService',
            message: 'No polyfill URI provided in chat service',
            package: 'tachyon-chat',
          });
        } else {
          try {
            logger.info({
              category: 'ChatService',
              message: `Polyfilling worker: ${message.data.payload.polyfillURI}`,
              package: 'tachyon-chat',
            });
            this.importScripts(polyfillURI);
          } catch (e) {
            // log error but continue, giving newer browsers a chance since all we need is fetch
            logger.warn({
              category: 'ChatService',
              context: {
                error: errorMessageFromCatch(e),
              },
              message: 'Error fetching polyfill in chat service',
              package: 'tachyon-chat',
            });
          }
        }

        this.channelLogin = channelLogin;
        this.channelID = channelID;

        if (!this.client) {
          this.client = new ChatClient(clientApiId);
        }

        return this.connect();
      case 'CHAT_WORKER_DISCONNECTED':
        return this.disconnect();
      case 'CHAT_WORKER_CHANNEL_CHANGED':
        this.channelLogin = message.data.payload.channelLogin;
        this.channelID = message.data.payload.channelID;

        return this.changeChannel();
      default:
        logger.error({
          category: 'ChatService',
          context: message.data,
          message: 'Unknown message type',
          package: 'tachyon-chat',
        });

        return Promise.resolve(undefined);
    }
  }

  public connect(): Promise<[void, void] | void> {
    if (!this.channelLogin || !this.channelID) {
      throw new Error(
        'Insufficient state to connect to chat. Need name and ID.',
      );
    }

    if (!this.client) {
      throw new Error('Cannot connect to chat, client not initialized');
    }

    logger.info({
      category: 'ChatService',
      message: `Instantiating client and connecting to ${this.channelLogin}`,
      package: 'tachyon-chat',
    });

    this.client.setConnectedHandler(this.onConnectedEvent);
    this.client.setDisconnectedHandler(this.onDisconnectedEvent);
    this.client.setReconnectHandler(this.onReconnectEvent);

    this.client.setHostingHandler(this.onHostingEvent);
    this.client.setUnhostHandler(this.onUnhostEvent);

    this.client.setChatHandler(this.onChatEvent('POST'));
    this.client.setActionHandler(this.onChatEvent('ACTION'));
    this.client.setTimeoutHandler(this.onModerationEvent('TIMEOUT'));
    this.client.setBanHandler(this.onModerationEvent('BAN'));
    this.client.setSubcriptionHandler(this.onSubscriptionEvent);
    this.client.setResubscriptionHandler(this.onResubscriptionEvent);

    return Promise.all([
      this.client.connect(this.channelLogin),
      this.badger.init(this.channelID),
    ]).catch((error) => {
      logger.error({
        category: 'ChatService',
        error,
        message: 'Error with initial connection to chat service',
        package: 'tachyon-chat',
      });
      if (this.client && !this.client.isSocketOpen()) {
        this.postMessage(reconnectEvent());
      }
    });
  }

  public disconnect(): Promise<void> {
    logger.info({
      category: 'ChatService',
      message: 'Disconnecting client and closing worker',
      package: 'tachyon-chat',
    });
    const promise = this.client ? this.client.disconnect() : Promise.resolve();
    return promise.then(() => {
      self.close();
    });
  }

  public changeChannel(): Promise<void> {
    if (!this.channelLogin || !this.channelID || !this.client) {
      throw new Error('Cannot change channel without channel name & ID');
    }

    if (!this.client) {
      throw new Error('Cannot change channel without initialized client');
    }

    logger.info({
      category: 'ChatService',
      message: `Changing channel connection to ${this.channelLogin}`,
      package: 'tachyon-chat',
    });

    return Promise.all([
      this.client.changeChannel(this.channelLogin),
      this.badger.init(this.channelID),
    ])
      .then(() => {
        this.onChannelChangeEvent();
      })
      .catch((error) => {
        logger.error({
          category: 'ChatService',
          error,
          message: 'Error with changing chat channel',
          package: 'tachyon-chat',
        });
      });
  }

  public onConnectedEvent = (_address: string, _port: number): void =>
    this.postMessage(connectedEvent());

  public onChannelChangeEvent = (): void => this.postMessage(connectedEvent());

  public onDisconnectedEvent = (reason: string): void =>
    this.postMessage(disconnectedEvent(reason));

  public onReconnectEvent = (): void => this.postMessage(reconnectEvent());

  public onHostingEvent =
    // TMI.js formats channel names like `#monstercat`
    (channel: string, target: string, _viewers: number): void =>
      this.postMessage(hostingEvent(channel.slice(1), target));

  public onUnhostEvent =
    // TMI.js formats channel names like `#monstercat`
    (channel: string, _viewers: number): void =>
      this.postMessage(unhostEvent(channel.slice(1)));

  public onChatEvent =
    (type: MessageEventType): ChatHandler =>
    (
      _channel: string,
      userstate: UserStatePayload,
      message: string,
      _sentByCurrentUser: boolean,
    ): void =>
      this.postMessage(
        messageEvent(
          type,
          message,
          userstate,
          this.badger.getBadgeData(userstate.badges),
        ),
      );

  public onModerationEvent =
    (type: ModerationEventType): ModerationHandler =>
    (
      _channel: string,
      username: string,
      reason: string,
      duration?: number,
    ): void =>
      this.postMessage(moderationEvent(type, username, reason, duration));

  public onSubscriptionEvent =
    // TMI.js formats channel names like `#monstercat`
    (channel: string, username: string, methods: SubscriptionMethods): void =>
      this.postMessage(
        subscriptionEvent(channel.slice(1), username, methods.prime),
      );

  public onResubscriptionEvent =
    // TMI.js formats channel names like `#monstercat`
    (
      channel: string,
      username: string,
      months: number,
      message: string,
      userstate: UserStatePayload,
      methods: SubscriptionMethods,
    ): void =>
      this.postMessage(
        resubscriptionEvent(
          channel.slice(1),
          username,
          methods.prime,
          months,
          message,
          userstate,
          userstate ? this.badger.getBadgeData(userstate.badges) : [],
        ),
      );
}
