import type { ChatEvent } from '../../events';
import type { WorkerMessage } from '../../types';
import { BadgerService } from '../BadgerService';
import { TEST_CHANNEL_ID, mockAllBadges } from '../BadgerService/test-utils';
import { ChatClient } from '../ChatClient';
import { ChatService } from '.';

jest.mock('../ChatClient');

jest.mock('tachyon-logger', () => ({
  logger: {
    error: jest.fn(),
    info: jest.fn(),
  },
}));

function getMockClient(): ChatClient | undefined {
  const {
    mock: {
      instances: [client],
    },
  } = ChatClient as jest.Mock;
  if (client) {
    client.disconnect = jest.fn().mockResolvedValue(undefined);
  }
  return client;
}

describe(ChatService, () => {
  let chatService: ChatService;
  let badger: BadgerService;
  let polyfillURI: string;

  function importScripts(uri: string): void {
    polyfillURI = uri;
  }

  function postMessage(_event: ChatEvent): void {
    return;
  }

  const originalClose = window.close;

  beforeAll(() => {
    window.close = jest.fn();
  });

  afterAll(() => {
    window.close = originalClose;
  });

  beforeEach(() => {
    (fetch as any).resetMocks();
    (ChatClient as jest.Mock).mockReset();
    badger = new BadgerService();
    chatService = new ChatService({ badger, importScripts, postMessage });
    mockAllBadges();
  });

  describe('messageHandler', () => {
    const CONNECT_MESSAGE: WorkerMessage = {
      command: 'CHAT_WORKER_CONNECTED',
      payload: {
        channelID: TEST_CHANNEL_ID,
        channelLogin: 'voxel',
        clientApiId: '12343254',
        polyfillURI: '//polyfill.sexy',
      },
    };

    it('connects on CHAT_WORKER_CONNECTED message', async () => {
      expect(getMockClient()).toBeUndefined();
      await chatService.messageHandler({ data: CONNECT_MESSAGE });

      expect(chatService.channelLogin).toEqual(
        CONNECT_MESSAGE.payload.channelLogin,
      );
      expect(chatService.channelID).toEqual(CONNECT_MESSAGE.payload.channelID);

      const mockClient = getMockClient();
      expect(mockClient!.connect).toHaveBeenCalledTimes(1);
      expect(mockClient!.connect).toHaveBeenCalledWith(
        CONNECT_MESSAGE.payload.channelLogin,
      );

      expect(mockClient!.setConnectedHandler).toHaveBeenCalledTimes(1);
      expect(mockClient!.setDisconnectedHandler).toHaveBeenCalledTimes(1);
      expect(mockClient!.setReconnectHandler).toHaveBeenCalledTimes(1);
      expect(mockClient!.setHostingHandler).toHaveBeenCalledTimes(1);
      expect(mockClient!.setUnhostHandler).toHaveBeenCalledTimes(1);
      expect(mockClient!.setChatHandler).toHaveBeenCalledTimes(1);
      expect(mockClient!.setActionHandler).toHaveBeenCalledTimes(1);
      expect(mockClient!.setTimeoutHandler).toHaveBeenCalledTimes(1);
      expect(mockClient!.setBanHandler).toHaveBeenCalledTimes(1);
      expect(mockClient!.setSubcriptionHandler).toHaveBeenCalledTimes(1);
      expect(mockClient!.setResubscriptionHandler).toHaveBeenCalledTimes(1);

      expect(badger.badges).toBeDefined();
      expect(badger.badges['warcraft']).toBeDefined();

      expect(polyfillURI).toEqual('//polyfill.sexy');
    });

    it('changes chat channel on CHAT_WORKER_CHANNEL_CHANGED message', async () => {
      mockAllBadges();
      const changeChannelMessage: WorkerMessage = {
        command: 'CHAT_WORKER_CHANNEL_CHANGED',
        payload: {
          channelID: TEST_CHANNEL_ID,
          channelLogin: 'monstercat',
        },
      };

      await chatService.messageHandler({ data: CONNECT_MESSAGE });

      expect(chatService.channelLogin).toEqual(
        CONNECT_MESSAGE.payload.channelLogin,
      );
      expect(chatService.channelID).toEqual(CONNECT_MESSAGE.payload.channelID);

      const mockClient = getMockClient();
      expect(mockClient!.connect).toHaveBeenCalledTimes(1);
      expect(mockClient!.connect).toHaveBeenCalledWith(
        CONNECT_MESSAGE.payload.channelLogin,
      );

      await chatService.messageHandler({ data: changeChannelMessage });

      expect(chatService.channelLogin).toEqual(
        changeChannelMessage.payload.channelLogin,
      );
      expect(chatService.channelID).toEqual(
        changeChannelMessage.payload.channelID,
      );

      expect(mockClient!.changeChannel).toHaveBeenCalledTimes(1);
      expect(mockClient!.changeChannel).toHaveBeenCalledWith(
        changeChannelMessage.payload.channelLogin,
      );
    });

    it('disconnects on CHAT_WORKER_DISCONNECTED message', async () => {
      const disconnectMessage: WorkerMessage = {
        command: 'CHAT_WORKER_DISCONNECTED',
      };

      await chatService.messageHandler({ data: CONNECT_MESSAGE });

      const mockClient = getMockClient();
      expect(mockClient!.disconnect).not.toHaveBeenCalled();

      await chatService.messageHandler({ data: disconnectMessage });
      expect(mockClient!.disconnect).toHaveBeenCalledTimes(1);
      expect(window.close).toHaveBeenCalledTimes(1);
    });
  });
});
