import { join } from 'path';
import { cwd } from 'process';
import { mkdirp, writeJsonSync } from 'fs-extra';
import type { MochaOptions } from 'mocha';
import { Runner } from 'mocha';

// TODO: move into common package
const successScreenshotName = 'Success-Screenshot';
// Reference: https://mochajs.org/api/tutorial-custom-reporter.html
export function resilienceReporter(
  runner: Runner,
  options?: MochaOptions,
): void {
  const { EVENT_RUN_END, EVENT_TEST_END } = Runner.constants;

  // overall report structure for a client run
  type ResilienceReport = {
    capabilityResults: Array<CapabilityReport>;
    /**
     * Overall client score = totalPassedScenarios/totalRanScenarios
     */
    clientResilienceScore: number;
  };

  type CapabilityReport = {
    /**
     * totalPassedScenariosForCapability/totalRanScenariosForCapability
     */
    capabilityResiliencyScore: number;
    /**
     * the name of the Capability ie. “Ability to make a living”
     */
    name: string;
    /**
     * aggregated data for each scenario
     */
    scenarioResults: Array<ScenarioReport>;
    /**
     * all service failures tested
     */
    servicesTested: Array<string>;
  };

  type ScenarioReport = {
    /**
     * % of failures handled (handled / handled + failures)
     */
    errorHandlingScore: number;
    /**
     * file name of what success looks like for this scenario
     */
    expectedScreenshot: string;
    /**
     * runs failed
     */
    failed: number;
    /**
     * detailed report of failures
     */
    failureInfos: Array<FailureReport>;
    /**
     * runs completed with an alternate path due to a failure
     */
    handledFailure: number;
    /**
     * the name of the scenario ie. “Open channel page”
     */
    name: string;
    /**
     * runs completed with the main or alternate path
     */
    passed: number;
    /**
     * total runs for that scenario
     */
    ran: number;
    /**
     * passed/ran
     */
    resiliencyScore: number;
    /**
     * all services tested for the given scenario
     */
    servicesTested: Array<string>;
  };

  type QueryReport = {
    /**
     * fields that could not be retrieved
     */
    errorFields: Array<string>;
    /**
     * the impacted GQL query
     */
    queryName: string;
  };

  type FailureReport = {
    /**
     * Array of queries involved
     */
    impactedQueries: Array<QueryReport>;
    /**
     * file name of the corresponding error screenshot
     */
    screenshot: string;
    /**
     * the simulated service failure
     */
    serviceFailed: string;
  };

  const resReport: ResilienceReport = {
    capabilityResults: [],
    clientResilienceScore: 0,
  };

  runner
    .on(EVENT_TEST_END, (test) => {
      // these should never be absent in a valid resilience test
      // this makes types more convenient
      if (!test.parent || !options) {
        return;
      }
      // displayInfo solely for test UI
      const [_displayInfo, currServiceName] = test.title.split(' ');
      const [suite, currServiceTest] = test.titlePath();
      // capability--feature--queryName
      const [capability, feature, queryName] = test.parent.title.split(
        '--',
      ) as [string, string, string];

      // If the capability currently being tested is
      // not in resReport then push the empty capability object
      if (!resReport.capabilityResults.find((x) => x.name === capability)) {
        resReport.capabilityResults.push({
          capabilityResiliencyScore: 0,
          name: capability,
          scenarioResults: [],
          servicesTested: [],
        });
      }
      const currCapability = resReport.capabilityResults.find(
        (x) => x.name === capability,
      );
      // If the capability object is present in resReport
      if (currCapability) {
        // If we are not looking at the successScreenshot test (to avoid pushing null service) and the service
        // is being tested for the first time for the specific capability add it
        if (
          test.title.startsWith('Failing') &&
          !currCapability.servicesTested.find((x) => x === currServiceName)
        ) {
          currCapability.servicesTested.push(currServiceName);
        }
        // If the scenario has not been tested yet for the specifc capability add the empty scenario object
        if (!currCapability?.scenarioResults.find((x) => x.name === feature)) {
          currCapability?.scenarioResults.push({
            errorHandlingScore: 0,
            expectedScreenshot: '',
            failed: 0,
            failureInfos: [],
            handledFailure: 0,
            name: feature,
            passed: 0,
            ran: 0,
            resiliencyScore: 0,
            servicesTested: [],
          });
        }

        const currScenario = currCapability?.scenarioResults.find(
          (x) => x.name === feature,
        );
        // If the scenario exists for the current capability
        if (feature && currScenario) {
          // Grab screenshot from the successscreenshot test
          if (test.title === successScreenshotName && options) {
            currScenario.expectedScreenshot = `${options.reporterOptions.screenshotsFolder}/${test.title}.png`;
          }
          // test title is Failing Service_Name
          const currService = currScenario?.servicesTested.find(
            (x) => x === currServiceName,
          );
          // If the test failed when the service went down
          if (test.state === 'failed') {
            currScenario.ran += 1;
            currScenario.failed += 1;
            if (!currService) {
              currScenario.servicesTested.push(currServiceName);
            }
            // Add the failureinfo object which contains the failed screenshot string

            const failureScreenshot = `${options.reporterOptions.screenshotsFolder}/${suite} -- ${currServiceTest} (failed).png`;
            currScenario.failureInfos.push({
              impactedQueries: [{ errorFields: [], queryName }],
              screenshot: failureScreenshot,
              serviceFailed: currServiceName,
            });
            // If the test passed (avoiding counting false positive of successScreenshot test) when the service went down
          } else if (
            test.state === 'passed' &&
            test.title.startsWith('Failing')
          ) {
            currScenario.ran += 1;
            currScenario.passed += 1;
            currScenario.servicesTested.push(currServiceName);
          }
        }
      }
    })

    .once(EVENT_RUN_END, () => {
      // following code is for calculating resiliency scores
      let totalPassed = 0;
      let totalRan = 0;
      for (
        let capabilityIndex = 0;
        capabilityIndex < resReport.capabilityResults.length;
        capabilityIndex++
      ) {
        let capabilityPassed = 0;
        let capabilityRan = 0;
        const currCapability = resReport.capabilityResults[capabilityIndex];
        for (
          let scenarioIndex = 0;
          scenarioIndex <
          resReport.capabilityResults[capabilityIndex].scenarioResults.length;
          scenarioIndex++
        ) {
          const currScenario = currCapability.scenarioResults[scenarioIndex];
          capabilityPassed += currScenario.passed;
          capabilityRan += currScenario.ran;
          // Calculate resiliency score for current scenario
          if (currScenario.ran > 0) {
            currScenario.resiliencyScore =
              currScenario.passed / currScenario.ran;
          }
        }
        // Calculate resiliency score for current capability
        if (capabilityRan > 0) {
          currCapability.capabilityResiliencyScore =
            capabilityPassed / capabilityRan;
        }
        totalPassed += capabilityPassed;
        totalRan += capabilityRan;
      }
      // Calculate clientResilienceScore
      if (totalRan > 0) {
        resReport.clientResilienceScore = totalPassed / totalRan;
      }
      // Directory and file to write resiliency data to
      const testDir = join(cwd(), 'test-results');
      mkdirp(testDir);
      writeJsonSync(join(testDir, 'resilience.json'), resReport);
    });
}
