import Bowser from 'bowser';
import type { TrackEvent } from 'tachyon-event-tracker';
import { useEffectOnce } from 'tachyon-utils-react';
import { isBrowser } from 'tachyon-utils-stdlib';
import { Enum } from 'tachyon-utils-ts';
import {
  BOWSER_USER_AGENT_FALLBACK,
  uniqueIDGenerator,
} from 'tachyon-utils-twitch';

interface DebugDeviceInfo {
  browser: string | undefined;
  browser_version: string | undefined;
  os: string | undefined;
  os_version: string | undefined;
}

export enum DebugType {
  LaserArrayError = 'lsr-error',
  TMWEnvironmentObservation = 'tmw-env',
  Test = 'test',
}

interface TachyonDebugBaseOpts {
  context: string;
  type: DebugType;
}

interface TachyonDebugOptionalOpts {
  number_value?: number;
  string_value?: string;
  subcontext?: string;
}

export interface TachyonDebugOpts
  extends TachyonDebugBaseOpts,
    TachyonDebugOptionalOpts {}

export interface TachyonDebugEvent extends DebugDeviceInfo, TachyonDebugOpts {
  debug_session_id: string;
  event: typeof DEBUG_EVENT_NAME;
}

interface DebugStoreForType {
  debugSessionId: string;
  reportedContexts: Record<string, true>;
  sessionWideValues: TachyonDebugOptionalOpts;
}

type DebugStore = { [k in DebugType]: DebugStoreForType };

const DEBUG_EVENT_NAME = 'mobile_web_debug';
const MAJOR_MINOR_REGEX = /^(\d+\.\d+)/;

// convert to tachyon-utils with test coverage
function convertToMajorMinorVersion(
  version: string | undefined,
): string | undefined {
  return version && MAJOR_MINOR_REGEX.test(version)
    ? version.match(MAJOR_MINOR_REGEX)![0] // eslint-disable-line @typescript-eslint/no-non-null-assertion
    : undefined;
}

function createNewDebugStoreEntryForType(): DebugStoreForType {
  return {
    debugSessionId: uniqueIDGenerator(),
    reportedContexts: {},
    sessionWideValues: {},
  };
}

// state
let bowserParser: Bowser.Parser.Parser | undefined;

const debugSessionStore: DebugStore = Enum.values(DebugType).reduce<DebugStore>(
  (acc, type) => {
    acc[type] = createNewDebugStoreEntryForType();

    return acc;
  },
  {} as DebugStore,
);

let deviceInfo: DebugDeviceInfo;

function setDeviceInfo(): void {
  // if in browser, generate bowser values
  if (isBrowser()) {
    bowserParser = Bowser.getParser(
      window.navigator.userAgent || BOWSER_USER_AGENT_FALLBACK,
    );

    const { name: os, version: osVersion } = bowserParser.getOS();
    const { name: browser, version: browserVersion } =
      bowserParser.getBrowser();

    deviceInfo = {
      browser,
      browser_version: convertToMajorMinorVersion(browserVersion),
      os,
      os_version: convertToMajorMinorVersion(osVersion),
    };
  } else {
    deviceInfo = {
      browser: undefined,
      browser_version: undefined,
      os: undefined,
      os_version: undefined,
    };
  }
}

let trackDebugEvent: TrackEvent<TachyonDebugEvent> =
  // eslint-disable-next-line no-console
  (event: TachyonDebugEvent) => console.log(event);

// exports
export function configTachyonDebug(
  reportEvent: TrackEvent<TachyonDebugEvent>,
): void {
  trackDebugEvent = reportEvent;
}

export function startNewTachyonDebugSession(type: DebugType): void {
  debugSessionStore[type] = createNewDebugStoreEntryForType();
}

export function getDebugSessionId(type: DebugType): string {
  return debugSessionStore[type].debugSessionId;
}

/**
 * Sets values for usage throughout a debug session. When the optional merge arg
 * is omitted or false, _all_ existing values will be removed and the new values
 * will be the only values persisted. If merge is true, then only the keys
 * found in the new values will clobber old values.
 */
export function setDebugSessionValues(
  type: DebugType,
  values: TachyonDebugOptionalOpts,
  merge = false,
): void {
  debugSessionStore[type].sessionWideValues = merge
    ? {
        ...debugSessionStore[type].sessionWideValues,
        ...values,
      }
    : {
        ...values,
      };
}

export function reportTachyonDebug(opts: TachyonDebugOpts): void {
  if (!deviceInfo) {
    setDeviceInfo();
  }

  if (debugSessionStore[opts.type].reportedContexts[opts.context]) {
    return;
  }

  debugSessionStore[opts.type].reportedContexts[opts.context] = true;

  const event: TachyonDebugEvent = {
    ...debugSessionStore[opts.type].sessionWideValues,
    ...opts,
    ...deviceInfo,
    debug_session_id: debugSessionStore[opts.type].debugSessionId,
    event: DEBUG_EVENT_NAME,
  };

  trackDebugEvent(event);
}

/**
 * React Component for reporting debug events in a declarative way. Will only
 * report once per render, since changing context is the only way to get around
 * the debug framework's report debouncing. Since context is considered the
 * top-level aggregator for debug events, it would not make sense to vary
 * context in a single location.
 */
export function TachyonDebug(props: TachyonDebugOpts): null {
  useEffectOnce(() => {
    reportTachyonDebug(props);
  });

  return null;
}
