import { random } from 'faker';
import { MockUserAgent } from 'tachyon-environment';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import type { TachyonDebugOpts } from '.';
import {
  DebugType,
  TachyonDebug,
  configTachyonDebug,
  reportTachyonDebug,
  setDebugSessionValues,
  startNewTachyonDebugSession,
} from '.';

describe('debug reporter', () => {
  // we have to fully clear jsdom's stub here
  delete (window as any).navigator;
  (window as any).navigator = {
    userAgent: MockUserAgent.iOS[12].iPhone,
  };

  function getRandomDebugData(): string {
    return random.alphaNumeric(24);
  }

  function getDebugEvent(): TachyonDebugOpts {
    return {
      context: getRandomDebugData(),
      type: DebugType.Test,
    };
  }

  const reportEvent = jest.fn();
  configTachyonDebug(reportEvent);

  beforeEach(() => {
    reportEvent.mockReset();
  });

  it('accepts config, reports events, does not double report, and updates debug session id and reports again', () => {
    const debugEvent = getDebugEvent();
    reportTachyonDebug(debugEvent);

    expect(reportEvent).toHaveBeenCalledWith({
      // get value from imported const once not a single file package
      ...debugEvent,
      browser: 'Safari',
      browser_version: '12.0',
      debug_session_id: expect.any(String),
      event: 'mobile_web_debug',
      os: 'iOS',
      os_version: '12.1',
    });

    reportTachyonDebug(debugEvent);
    expect(reportEvent).toHaveBeenCalledTimes(1);

    startNewTachyonDebugSession(DebugType.Test);
    reportTachyonDebug(debugEvent);

    expect(reportEvent.mock.calls[0][0].debug_session_id).not.toEqual(
      reportEvent.mock.calls[1][0].debug_session_id,
    );
  });

  describe(setDebugSessionValues, () => {
    it('accepts session-wide debug values', () => {
      const subcontext = getRandomDebugData();
      setDebugSessionValues(DebugType.Test, { subcontext });
      reportTachyonDebug(getDebugEvent());

      expect(reportEvent).toHaveBeenCalledWith(
        expect.objectContaining({
          event: 'mobile_web_debug',
          subcontext,
        }),
      );
    });

    it('clobbers values from multiple calls when merge is false', () => {
      const subcontext = getRandomDebugData();
      const string_value = getRandomDebugData();
      setDebugSessionValues(DebugType.Test, { subcontext });
      setDebugSessionValues(DebugType.Test, { string_value });
      reportTachyonDebug(getDebugEvent());

      expect(reportEvent).toHaveBeenCalledWith(
        expect.objectContaining({
          event: 'mobile_web_debug',
          string_value,
        }),
      );
      expect(reportEvent).toHaveBeenCalledWith(
        expect.not.objectContaining({
          subcontext: expect.anything(),
        }),
      );
    });

    it('merges values from previous calls when merge is true', () => {
      const subcontext = getRandomDebugData();
      const string_value = getRandomDebugData();
      setDebugSessionValues(DebugType.Test, { subcontext });
      setDebugSessionValues(DebugType.Test, { string_value }, true);
      reportTachyonDebug(getDebugEvent());

      expect(reportEvent).toHaveBeenCalledWith(
        expect.objectContaining({
          event: 'mobile_web_debug',
          string_value,
          subcontext,
        }),
      );
    });
  });

  describe('declarative reporting element', () => {
    const setup = createMountWrapperFactory(TachyonDebug, getDebugEvent);

    it('is empty render and reports only once even if context changes', () => {
      const { wrapper } = setup();
      expect(wrapper.html()).toBeNull();
      expect(reportEvent).toHaveBeenCalledTimes(1);

      wrapper.update();

      expect(reportEvent).toHaveBeenCalledTimes(1);

      wrapper.setProps({ context: getRandomDebugData() });

      expect(reportEvent).toHaveBeenCalledTimes(1);
    });
  });
});
