import type {
  DynamicSettings,
  DynamicSettingsAppProcessor,
  DynamicSettingsBase,
  DynamicSettingsRawData,
  DynamicSettingsRawSpadeData,
} from '../types';

/**
 * The fallback environment value
 */
export const ENVIRONMENT_FALLBACK = 'production';
/**
 * The fallback experiments value
 */
export const EXPERIMENTS_FALLBACK = {};
/**
 * The fallback URI when a Spade URI cannot be retrieved.
 */
export const SPADE_URL_FALLBACK = 'https://spade.twitch.tv/';

export const DYNAMIC_SETTINGS_FALLBACK: DynamicSettingsBase = {
  environment: ENVIRONMENT_FALLBACK,
  experiments: EXPERIMENTS_FALLBACK,
  /**
   * The fallback URI when a Spade url cannot be retrieved.
   */
  spadeUrl: SPADE_URL_FALLBACK,
};

// protect against missing or invalid data returned for this value and force https
export function ensureValidSecureSpadeUrl(url: string | undefined): string {
  if (!url) {
    return SPADE_URL_FALLBACK;
  }

  const protocolUrl = /^\/\//.test(url) ? `https:${url}` : url;
  try {
    const validatedUrl = new URL(protocolUrl);
    validatedUrl.protocol = 'https';
    return validatedUrl.toString();
  } catch {
    return SPADE_URL_FALLBACK;
  }
}

export function dynamicsSettingsResponseProcessor<
  RawCustomDynamicSettings extends {},
>(
  appProcessor: DynamicSettingsAppProcessor<RawCustomDynamicSettings>,
): (
  resp: DynamicSettingsRawSpadeData<RawCustomDynamicSettings>,
) => DynamicSettings<RawCustomDynamicSettings> {
  return (
    resp: DynamicSettingsRawSpadeData<RawCustomDynamicSettings>,
  ): DynamicSettings<RawCustomDynamicSettings> => {
    const { environment, experiments, spade_url, ...rawData } = resp;

    const baseData = {
      ...rawData,
      environment: environment ?? ENVIRONMENT_FALLBACK,
      experiments: experiments ?? EXPERIMENTS_FALLBACK,
      spadeUrl: ensureValidSecureSpadeUrl(spade_url),
    } as DynamicSettingsRawData<RawCustomDynamicSettings>;

    return appProcessor(baseData);
  };
}
