import { datatype, internet } from 'faker';
import type {
  DynamicSettings,
  DynamicSettingsAppProcessor,
  DynamicSettingsRawSpadeData,
} from '../types';
import {
  ENVIRONMENT_FALLBACK,
  EXPERIMENTS_FALLBACK,
  SPADE_URL_FALLBACK,
  dynamicsSettingsResponseProcessor,
  ensureValidSecureSpadeUrl,
} from '.';

describe('dynamic settings response processor', () => {
  describe(dynamicsSettingsResponseProcessor, () => {
    type RawCustomDynamicSettings = { a_b: number };

    const processor: DynamicSettingsAppProcessor<RawCustomDynamicSettings> = ({
      a_b,
      ...base
    }) => ({
      ...base,
      aB: a_b as number,
    });

    it('converts base data and applies app processor function', () => {
      const resp: DynamicSettingsRawSpadeData<RawCustomDynamicSettings> = {
        a_b: datatype.number({ min: 1 }),
        environment: 'development',
        experiments: { realExperimentData: true } as any,
        spade_url: internet.url().replace('http:', 'https:'),
      };

      const expectedOutput: DynamicSettings<RawCustomDynamicSettings> = {
        aB: resp.a_b!,
        environment: resp.environment!,
        experiments: resp.experiments!,
        spadeUrl: `${resp.spade_url}/`,
      };

      expect(dynamicsSettingsResponseProcessor(processor)(resp)).toEqual(
        expectedOutput,
      );
    });

    it('normalizes missing base data and applies app processor function', () => {
      const resp: DynamicSettingsRawSpadeData<RawCustomDynamicSettings> = {
        a_b: datatype.number({ min: 1 }),
      };

      const expectedOutput: DynamicSettings<RawCustomDynamicSettings> = {
        aB: resp.a_b!,
        environment: ENVIRONMENT_FALLBACK,
        experiments: EXPERIMENTS_FALLBACK,
        spadeUrl: SPADE_URL_FALLBACK,
      };

      expect(dynamicsSettingsResponseProcessor(processor)(resp)).toEqual(
        expectedOutput,
      );
    });
  });

  describe(ensureValidSecureSpadeUrl, () => {
    it('returns fallback if url is undefined', () => {
      expect(ensureValidSecureSpadeUrl(undefined)).toEqual(SPADE_URL_FALLBACK);
    });

    it('prepends https to protocol relative URLs', () => {
      expect(ensureValidSecureSpadeUrl('//yolo/')).toEqual('https://yolo/');
    });

    it('appends slashes as natural behavior of WHATWG URL', () => {
      expect(ensureValidSecureSpadeUrl('https://yolo')).toEqual(
        'https://yolo/',
      );
    });

    it('does nothing to https URLs', () => {
      expect(ensureValidSecureSpadeUrl('https://yolo/')).toEqual(
        'https://yolo/',
      );
    });

    it('upgrades http URLs', () => {
      expect(ensureValidSecureSpadeUrl('http://yolo/')).toEqual(
        'https://yolo/',
      );
    });

    it('does not throw on malformed URI and instead returns fallback', () => {
      expect(() => {
        expect(ensureValidSecureSpadeUrl('bad.url')).toEqual(
          SPADE_URL_FALLBACK,
        );
      }).not.toThrow();
    });
  });
});
