import { logger } from 'tachyon-logger';
import {
  Formats,
  errorMessageFromCatch,
  fetchData,
} from 'tachyon-utils-stdlib';
import {
  DYNAMIC_SETTINGS_FALLBACK,
  dynamicsSettingsResponseProcessor,
} from '../dynamicSettingsResponseProcessor';
import type { DynamicSettingsURLGeneratorOpts } from '../dynamicSettingsURLGenerator';
import { dynamicSettingsURLGenerator } from '../dynamicSettingsURLGenerator';
import type {
  DynamicSettings,
  DynamicSettingsAppProcessor,
  DynamicSettingsRawData,
  DynamicSettingsRawSpadeData,
} from '../types';

export type FetchDynamicSettingsOpts<RawCustomDynamicSettings extends {}> =
  DynamicSettingsURLGeneratorOpts & {
    /**
     * Function for transforming the raw dynamic settings response into the shape
     * used in the app. This includes not only converting from snake_case raw
     * settings to camelCase processed settings (both enforced by types), but also
     * presence and shape verification (all values are treated as unknown to
     * protect against uncoordinated changes to savant/actuator values).
     */
    processor: DynamicSettingsAppProcessor<RawCustomDynamicSettings>;
  };

export function fetchDynamicSettings<RawCustomDynamicSettings extends {}>({
  app,
  appEnvironment,
  appGroup,
  processor,
}: FetchDynamicSettingsOpts<RawCustomDynamicSettings>): () => Promise<
  DynamicSettings<RawCustomDynamicSettings>
> {
  const url = dynamicSettingsURLGenerator({ app, appEnvironment, appGroup });
  const responseProcessor = dynamicsSettingsResponseProcessor(processor);

  return async () => {
    try {
      const res = await fetchData<
        DynamicSettingsRawSpadeData<RawCustomDynamicSettings>
      >(url, { Accept: 'application/json' }, Formats.JSON);
      return responseProcessor(res);
    } catch (e) {
      logger.warn({
        category: 'fetchDynamicSettings',
        context: {
          url,
        },
        message: errorMessageFromCatch(e),
        package: 'dynamic-settings',
      });

      // we process the fallback to allow the app to populate fallbacks for its
      // custom properties also
      return processor(
        DYNAMIC_SETTINGS_FALLBACK as DynamicSettingsRawData<RawCustomDynamicSettings>,
      );
    }
  };
}
