import { lorem } from 'faker';
import { logger } from 'tachyon-logger';
import { Formats, fetchData } from 'tachyon-utils-stdlib';
import { DYNAMIC_SETTINGS_FALLBACK } from '..';
import type { FetchDynamicSettingsOpts } from '.';
import { fetchDynamicSettings } from '.';

jest.mock('tachyon-utils-stdlib', () => ({
  ...jest.requireActual('tachyon-utils-stdlib'),
  fetchData: jest.fn(),
}));
const mockFetchData = fetchData as jest.Mock;

jest.mock('tachyon-logger', () => ({
  logger: { warn: jest.fn() },
}));
const mockLoggerWarn = logger.warn as jest.Mock;

describe(fetchDynamicSettings, () => {
  const URL = 'https://static.twitchcdn.net/config/settings.tachyon.app.json';

  type Setup = {
    fetcher: () => Promise<{}>;
    processor: jest.Mock;
  };

  function setup(options: Partial<FetchDynamicSettingsOpts<{}>> = {}): Setup {
    const processor = jest.fn((data) => data);
    const fetcher = fetchDynamicSettings({
      app: 'app',
      appEnvironment: 'production',
      appGroup: 'tachyon',
      processor,
      ...options,
    });

    return {
      fetcher,
      processor,
    };
  }

  afterEach(() => {
    mockFetchData.mockReset();
    mockLoggerWarn.mockReset();
  });

  describe('the resultant with a successful response', () => {
    const response = { a: 1 };
    beforeEach(() => {
      mockFetchData.mockImplementation(() => response);
    });

    it('makes an initial request with all the right parameters and processes data', async () => {
      const { fetcher, processor } = setup();

      const expected = {
        ...response,
        environment: 'production',
        experiments: {},
        spadeUrl: 'https://spade.twitch.tv/',
      };
      expect(await fetcher()).toEqual(expected);
      expect(mockFetchData).toHaveBeenCalledTimes(1);
      expect(mockFetchData).toHaveBeenCalledWith(
        URL,
        { Accept: 'application/json' },
        Formats.JSON,
      );
      expect(processor).toHaveBeenCalledWith(expected);
    });
  });

  describe('the resultant with a request error', () => {
    const errorMessage = lorem.words(10);

    beforeEach(() => {
      mockFetchData.mockImplementation(() => {
        throw new Error(errorMessage);
      });
    });

    it('responds with the processed default and logs', async () => {
      const { fetcher, processor } = setup();
      expect(await fetcher()).toEqual(DYNAMIC_SETTINGS_FALLBACK);
      expect(mockFetchData).toHaveBeenCalledTimes(1);
      expect(processor).toHaveBeenCalledWith(DYNAMIC_SETTINGS_FALLBACK);

      expect(mockLoggerWarn).toHaveBeenCalledWith({
        category: 'fetchDynamicSettings',
        context: {
          url: URL,
        },
        message: expect.stringContaining(errorMessage),
        package: 'dynamic-settings',
      });
    });
  });
});
