# Tachyon Environment

A package that makes consuming client environment information simple in
React-based applications.

- SSR compatible
- Support for looking up a user's/client's: Device ID, Browser / OS, Network
  stats
- Supports network online/offline detection

## Installation

```sh
$ yarn add tachyon-environment
```

## Examples

- [Accessing Environment Info](../../../package-examples/client-environment-info/src/EnvironmentExample.tsx)
- [Checking OS/Browser Info](../../../package-examples/client-environment-info/src/ClientBrowserAndOSExample.tsx)
- [Checking Network Info](../../../package-examples/client-environment-info/src/ClientNetworkStatsExample.tsx)

## How To Consume

1. Use `EnvironmentRoot` at the top of your application hierarchy to initialize
   the system:

This package works both with SSR and non-SSR applications.

```tsx
import type { FC } from 'react';
import { EnvironmentRoot } from 'tachyon-environment';

const AppRoot: FC = () => (
  <EnvironmentRoot common={{...}}>
    {/* application components */}
  </EnvironmentRoot>
);
```

2. Inject computed environment info into a component:

> using the `useStaticEnvironment` hook:

```tsx
import type { FC } from 'react';
import { OS, useStaticEnvironment } from 'tachyon-environment';

export const PlatformSpecificFeature: FC = () => {
  const { common: { platform } } = useStaticEnvironment();
  if (platform === Platform.Unknown) {
    return null;
  }

  return (
    // ...
  );
}
```

> using the `withStaticEnvironment` HOC:

```tsx
import { Component } from 'react';
import { StaticEnvironmentProps, OS, withStaticEnvironment } from 'tachyon-environment';

class AndroidOnlyFeatureBase extends Component<EnvironmentProps> {
  public override render(): JSX.Element | null {
    const { os } = this.props.staticEnv.client;
    if (os !== OS.Android) {
      return null;
    }

    return (
      // ...
    );
  }
}

export const AndroidOnlyFeature = withStaticEnvironment(AndroidOnlyFeatureBase);
```

### Network Status Detection

This package also provides an API for consuming network status via
[offline and online](https://developer.mozilla.org/en-US/docs/Web/API/Navigator/Online_and_offline_events)
events.

```tsx
import type { FC } from 'react';
import { OS, useDynamicEnviroment } from 'tachyon-environment';

export const OfflineOnlyFeature: FC = () => {
  const { isOffline } = useDynamicEnviroment();
  if (!isOffline) {
    return null;
  }

  return (
    // ...
  );
}
```

## Testing Components

### Mocking The `useStaticEnvironment` Hook In Tests

This package includes a mocking function for the `useStaticEnvironment` hook in
tests:

```tsx
jest.mock('tachyon-environment', () => {
  const original = require.requireActual('tachyon-environment');
  return {
    ...original,
    useStaticEnvironment: original.mockUseStaticEnvironment(),
  };
  // mockUseStaticEnvironment() will default to mocking the iOS environment.
});
```

If you need a specific environment you can pass it to `mockUseStaticEnvironment`
like so:

```tsx
jest.mock('tachyon-environment', () => {
  const original = require.requireActual('tachyon-environment');
  const mockedAndroidEnv = original.getAndroidEnvMock();
  return {
    ...original,
    useStaticEnvironment: original.mockUseStaticEnvironment(mockedAndroidEnv),
  };
});
```

### Mocking Environment In Tests

This package also includes mock environment objects that you can use to simulate
various environments in tests:

```tsx
import { getAndroidEnvMock } from 'tachyon-environment';

[...]
  <SomeComponentBase env={getAndroidEnvMock()} />
[...]
```
