import type { FC } from 'react';
import { createContext, useEffect, useMemo, useRef, useState } from 'react';
import type { DataScienceNetworkInformation } from 'tachyon-type-library';
import { useConst } from 'tachyon-utils-react';
import { isBrowser, isOffline } from 'tachyon-utils-stdlib';
import {
  getDeviceIDInBrowser,
  getOrCreateSessionID,
} from 'tachyon-utils-twitch';
import { Platform } from '../Platform';
import type { AgentInfo } from '../utils';
import { getAgentInfo, getNetInfo } from '../utils';

/**
 * Client environment values that are calculated from user-agent, cookies, APIs,
 * etc.
 */
export type ClientAmbientEnvironment = {
  /**
   * Information about the OS, Browser, Platform, and Engine detected from
   * the client's UserAgent.
   *
   * For more details, see: https://github.com/lancedikson/bowser#filtering-browsers
   */
  readonly agentInfo: AgentInfo;
  /**
   * Standard Twitch device ID, stored in cookie
   */
  readonly deviceID: string;
  /**
   * Info from the NetworkInformation API, if available
   */
  readonly netInfo: Readonly<DataScienceNetworkInformation>;
  /**
   * ID for the current browser session, stored in sessionStorage
   */
  readonly sessionID: string;
};

/**
 * Recommended query param key for communicating launcher version via URL
 * query params
 */
export const LAUNCHER_VERSION_QUERY_PARAM_KEY = 'lnchv';

/**
 * Client environment values that are passed in by the application. These values
 * should not be dynamic, and as such are cached on first read for performance.
 */
export type ClientConfiguredEnvironment = {
  /**
   * Version of the launcher used to, if any and known
   */
  readonly launcherVersion?: string;
};

/**
 * Environment values that are specific to the client/user
 */
export type ClientEnvironment = ClientAmbientEnvironment &
  ClientConfiguredEnvironment;

/**
 * Environment of the current running application instance.
 */
export type AppEnvironment = 'development' | 'production' | 'staging';

/**
 * Environment values that are specific to the application, common to all
 * clients/users. These values should not be dynamic, and as such are cached on
 * first read for performance.
 */
export type CommonEnvironment = {
  appEnvironment: AppEnvironment;
  appVersion: string;
  clientApp: string;
  language: string;
  platform: Platform;
};

/**
 * The publicly provided static environment data.
 */
export type StaticEnvironmentContext = {
  /**
   * Client subset of the environment data, usually only available after first mount
   * of the application.
   */
  readonly client: ClientEnvironment;
  /**
   * Common subset of the environment data, always available.
   */
  readonly common: CommonEnvironment;
};

export const defaultStaticEnvironmentContext: StaticEnvironmentContext = {
  client: {
    agentInfo: getAgentInfo(''),
    deviceID: '',
    netInfo: {} as DataScienceNetworkInformation,
    sessionID: '',
  },
  common: {
    appEnvironment: 'development',
    appVersion: '',
    clientApp: '',
    language: '',
    platform: Platform.Unknown,
  },
};

/**
 * Private context used within this package.
 */
export const staticEnvironmentContext = createContext<StaticEnvironmentContext>(
  defaultStaticEnvironmentContext,
);

/**
 * The shape of the dynamic environment; for components that can change over
 * time, unlike those in StaticEnvironment.
 */
export type DynamicEnvironmentContext = {
  /**
   * Whether or not the navigator is online.
   *
   * Browsers implement this property differently.
   *
   * In Chrome and Safari, if the browser is not able to connect to a local area network (LAN) or a router, it is offline; all other conditions return true. So while you can assume that the browser is offline when it returns a false value, you cannot assume that a true value necessarily means that the browser can access the internet.
   *
   * https://developer.mozilla.org/en-US/docs/Web/API/NavigatorOnLine/onLine
   */
  isOffline: boolean;
};

export const defaultDynamicEnvironmentContext: DynamicEnvironmentContext = {
  isOffline: true,
};

/**
 * Private context used within this package.
 */
export const dynamicEnvironmentContext =
  createContext<DynamicEnvironmentContext>(defaultDynamicEnvironmentContext);

export type EnvironmentRootProps = {
  /**
   * Environment values custom to a specific app.
   */
  client?: ClientConfiguredEnvironment;
  /**
   * Environment values that are agnostic to server or client-side requirements.
   */
  common: Omit<CommonEnvironment, 'platform'> & {
    /**
     * Either a Platform value or a function that takes AgentInfo and returns
     * a Platform value.
     */
    platform: Platform | ((agentInfo: AgentInfo) => Platform);
  };
  /**
   * In milliseconds, specifies the amount of time to delay network status changes.
   * It is used to prevent flapping while a device is experiencing poor network connection.
   * Setting this to null disables network status detection and sets isOffline
   * permanently to false.
   */
  networkStatusDebounceWait: number | null;
};

/**
 * EnvironmentRoot initializes an environment system for a React application,
 * and must be put in the React tree above all other environment functionality
 * from this package. It controls gating the gathering and release of the
 * client-specific environment data. This enables apps with SSR to ensure a
 * proper tree comparison during the first render followed by an update to
 * propagate the client data.
 */

export const EnvironmentRoot: FC<EnvironmentRootProps> = ({
  children,
  client = {},
  common,
  networkStatusDebounceWait,
}) => {
  const clientEnvironment = useConst(() =>
    isBrowser()
      ? {
          ...client,
          agentInfo: getAgentInfo(window.navigator.userAgent),
          deviceID: getDeviceIDInBrowser() || '',
          netInfo: getNetInfo(),
          sessionID: getOrCreateSessionID(),
        }
      : defaultStaticEnvironmentContext.client,
  );

  const commonEnvironment = useConst(() => {
    const platform = common.platform
      ? typeof common.platform === 'string'
        ? common.platform
        : common.platform(clientEnvironment.agentInfo)
      : Platform.Unknown;

    return {
      ...common,
      platform,
    };
  });

  const [offline, setOffline] = useState(() =>
    networkStatusDebounceWait === null ? false : isOffline(),
  );

  const networkChangeTimeoutIdRef = useRef<number>();

  useEffect(() => {
    if (networkStatusDebounceWait !== null) {
      const networkStatusChangeHandler = (event: Event) => {
        if (networkChangeTimeoutIdRef.current) {
          // If a network status change occurs, we want to reset our state change
          // timer regardless of what was pending to account for flapping
          window.clearTimeout(networkChangeTimeoutIdRef.current);
          networkChangeTimeoutIdRef.current = undefined;
        }

        networkChangeTimeoutIdRef.current = window.setTimeout(() => {
          setOffline(event.type === 'offline');
        }, networkStatusDebounceWait);
      };

      window.addEventListener('online', networkStatusChangeHandler);
      window.addEventListener('offline', networkStatusChangeHandler);

      return () => {
        window.removeEventListener('online', networkStatusChangeHandler);
        window.removeEventListener('offline', networkStatusChangeHandler);
        window.clearTimeout(networkChangeTimeoutIdRef.current);
      };
    }
  }, [networkStatusDebounceWait, setOffline]);

  const staticContext: StaticEnvironmentContext = useMemo(
    () => ({
      client: clientEnvironment,
      common: commonEnvironment,
    }),
    [clientEnvironment, commonEnvironment],
  );

  const dynamicContext: DynamicEnvironmentContext = useMemo(
    () => ({
      isOffline: offline,
    }),
    [offline],
  );

  return (
    <staticEnvironmentContext.Provider value={staticContext}>
      <dynamicEnvironmentContext.Provider
        children={children}
        value={dynamicContext}
      />
    </staticEnvironmentContext.Provider>
  );
};

EnvironmentRoot.displayName = 'EnvironmentRoot';
