import Bowser from 'bowser';
import { datatype, lorem, random } from 'faker';
import { act } from 'react-dom/test-utils';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import type { NetworkInformation } from 'tachyon-type-library';
import { ConnectionType, EffectiveConnectionType } from 'tachyon-type-library';
import { isBrowser, isOffline } from 'tachyon-utils-stdlib';
import {
  BOWSER_USER_AGENT_FALLBACK,
  getDeviceIDInBrowser,
  getOrCreateSessionID,
} from 'tachyon-utils-twitch';
import { Platform } from '../Platform';
import { getNetInfo } from '../utils';
import type {
  ClientConfiguredEnvironment,
  DynamicEnvironmentContext,
  StaticEnvironmentContext,
} from '.';
import {
  EnvironmentRoot,
  dynamicEnvironmentContext,
  staticEnvironmentContext,
} from '.';

jest.mock('tachyon-utils-stdlib', () => ({
  ...jest.requireActual('tachyon-utils-stdlib'),
  isBrowser: jest.fn(() => true),
  isOffline: jest.fn(() => false),
}));
const mockIsBrowser = isBrowser as jest.Mock;
const mockIsOffline = isOffline as jest.Mock;

jest.mock('tachyon-utils-twitch', () => ({
  ...jest.requireActual('tachyon-utils-twitch'),
  getDeviceIDInBrowser: jest.fn(),
  getOrCreateSessionID: jest.fn(),
}));
const mockGetDeviceIDInBrowser = getDeviceIDInBrowser as jest.Mock;
const mockGetOrCreateSessionID = getOrCreateSessionID as jest.Mock;

describe(EnvironmentRoot, () => {
  const maxNetworkStatusDebounceWait = 200;
  const setup = createMountWrapperFactory(EnvironmentRoot, () => ({
    children: (
      <staticEnvironmentContext.Consumer
        children={(staticEnv) => (
          <dynamicEnvironmentContext.Consumer
            children={(dynamicEnv) => {
              // render a div so we can inspect its props
              // lowercase prop names to avoid React warnings
              // @ts-expect-error: tests
              return <div dynamicenv={dynamicEnv} staticenv={staticEnv} />;
            }}
          />
        )}
      />
    ),
    common: {
      appEnvironment: 'development',
      appVersion: datatype.uuid(),
      clientApp: lorem.word(),
      language: random.locale(),
      platform: Platform.Unknown,
    },
    networkStatusDebounceWait: datatype.number({
      max: maxNetworkStatusDebounceWait,
      min: 10,
    }),
  }));

  type TestContexts = {
    dynamicEnv: DynamicEnvironmentContext;
    staticEnv: StaticEnvironmentContext;
  };

  function getContexts(
    wrapper: ReturnType<typeof setup>['wrapper'],
  ): TestContexts {
    const { dynamicenv: dynamicEnv, staticenv: staticEnv } = wrapper
      .find('div')
      .props() as any;
    return { dynamicEnv, staticEnv };
  }

  describe('when rendered in a server context', () => {
    it('renders a provider with empty client env values and default dynamic values', () => {
      mockIsBrowser.mockImplementationOnce(() => false);

      const { props, wrapper } = setup();
      const { dynamicEnv, staticEnv } = getContexts(wrapper);

      expect(staticEnv).toMatchObject({
        client: {
          deviceID: '',
          netInfo: {},
          sessionID: '',
        },
        common: props.common,
      });
      expect(staticEnv.client.agentInfo!.getResult()).toEqual(
        Bowser.parse(BOWSER_USER_AGENT_FALLBACK),
      );

      expect(dynamicEnv).toMatchObject({
        isOffline: false,
      });
    });
  });

  describe('when rendered in a browser context', () => {
    it('populates client env with actual info', () => {
      const deviceID = datatype.uuid();
      mockGetDeviceIDInBrowser.mockImplementationOnce(() => deviceID);
      const sessionID = datatype.uuid();
      mockGetOrCreateSessionID.mockImplementationOnce(() => sessionID);
      const isOfflineVal = datatype.boolean();
      mockIsOffline.mockImplementationOnce(() => isOfflineVal);

      const { props, wrapper } = setup();
      const { dynamicEnv, staticEnv } = getContexts(wrapper);

      expect(staticEnv).toMatchObject({
        client: {
          deviceID,
          netInfo: getNetInfo(),
          sessionID,
        },
        common: props.common,
      });
      expect(staticEnv.client.agentInfo!.getResult()).toEqual(
        Bowser.parse(window.navigator.userAgent),
      );

      expect(dynamicEnv).toMatchObject({
        isOffline: isOfflineVal,
      });
    });

    it('works with dynamic platform value', () => {
      const platform = jest.fn().mockReturnValue(Platform.StarshotStaging);
      const { wrapper } = setup({ common: { platform } });
      const { staticEnv } = getContexts(wrapper);

      expect(staticEnv.common.platform).toEqual(Platform.StarshotStaging);
      expect(platform).toHaveBeenCalledWith(staticEnv.client.agentInfo);
    });

    it('works with empty platform value', () => {
      const { wrapper } = setup({ common: { platform: '' as any } });
      const { staticEnv } = getContexts(wrapper);

      expect(staticEnv.common.platform).toEqual(Platform.Unknown);
    });

    it('works with configured client values', () => {
      const clientConfig: ClientConfiguredEnvironment = {
        launcherVersion: datatype.uuid(),
      };
      const { wrapper } = setup({ client: clientConfig });
      const { staticEnv } = getContexts(wrapper);

      expect(staticEnv.client).toEqual(expect.objectContaining(clientConfig));
    });
  });

  describe('online/offline detection', () => {
    const networkInformation: NetworkInformation = {
      downlink: datatype.number(),
      downlinkMax: datatype.number(),
      effectiveType: EffectiveConnectionType.SlowSecondGeneration,
      rtt: datatype.number(),
      type: ConnectionType.Cellular,
    };

    beforeEach(() => {
      navigator.connection = networkInformation;
      jest.clearAllTimers();
    });

    afterEach(() => {
      // @ts-expect-error: tests
      delete navigator.connection;
    });

    const ONLINE_EVENT = new Event('online');
    const OFFLINE_EVENT = new Event('offline');

    it('changes with a specific delay when an online or offline event is fired', () => {
      const { props, wrapper } = setup();

      expect(getContexts(wrapper).dynamicEnv.isOffline).toEqual(false);

      act(() => {
        window.dispatchEvent(OFFLINE_EVENT);
      });
      wrapper.update();
      expect(getContexts(wrapper).dynamicEnv.isOffline).toEqual(false);

      act(() => {
        jest.advanceTimersByTime(props.networkStatusDebounceWait!);
      });
      wrapper.update();
      expect(getContexts(wrapper).dynamicEnv.isOffline).toEqual(true);

      act(() => {
        window.dispatchEvent(ONLINE_EVENT);
      });
      wrapper.update();
      expect(getContexts(wrapper).dynamicEnv.isOffline).toEqual(true);

      act(() => {
        jest.advanceTimersByTime(props.networkStatusDebounceWait!);
      });
      wrapper.update();
      expect(getContexts(wrapper).dynamicEnv.isOffline).toEqual(false);
    });

    it('retains state during a flapping event', () => {
      const { props, wrapper } = setup();
      expect(getContexts(wrapper).dynamicEnv.isOffline).toEqual(false);

      act(() => {
        window.dispatchEvent(OFFLINE_EVENT);
      });
      wrapper.update();
      expect(getContexts(wrapper).dynamicEnv.isOffline).toEqual(false);

      act(() => {
        jest.advanceTimersByTime(props.networkStatusDebounceWait! - 1);
        window.dispatchEvent(ONLINE_EVENT);
      });
      wrapper.update();
      expect(getContexts(wrapper).dynamicEnv.isOffline).toEqual(false);

      act(() => {
        jest.advanceTimersByTime(props.networkStatusDebounceWait!);
      });
      wrapper.update();
      expect(getContexts(wrapper).dynamicEnv.isOffline).toEqual(false);
    });

    it('is disabled when networkStatusDebounceWait set to null', () => {
      const { wrapper } = setup({
        networkStatusDebounceWait: null,
      });

      expect(getContexts(wrapper).dynamicEnv.isOffline).toEqual(false);

      act(() => {
        window.dispatchEvent(OFFLINE_EVENT);
      });
      wrapper.update();
      expect(getContexts(wrapper).dynamicEnv.isOffline).toEqual(false);

      act(() => {
        jest.advanceTimersByTime(maxNetworkStatusDebounceWait + 1);
      });
      wrapper.update();
      expect(getContexts(wrapper).dynamicEnv.isOffline).toEqual(false);
    });
  });
});
