import { ConnectionType, EffectiveConnectionType } from 'tachyon-type-library';
import type {
  CommonEnvironment,
  StaticEnvironmentContext,
} from './EnvironmentRoot';
import { staticEnvironmentContext } from './EnvironmentRoot';
import { Platform } from './Platform';
import { getAgentInfo } from './utils';

// istanbul ignore file: test mocks

/**
 * Mock version of useStaticEnvironment for tests. Optionally accepts an env
 * target; otherwise defaults to iOS.
 */
export function mockUseStaticEnvironment(
  staticEnvValue: StaticEnvironmentContext = getiOSEnvMock(),
): () => StaticEnvironmentContext {
  return (): StaticEnvironmentContext => staticEnvValue;
}

/**
 * @experimental
 * For generating tuples compatible with tachyon-test-utils mocking functionality
 */
export function createMockStaticEnvironmentContext(
  staticEnvValue: StaticEnvironmentContext = getiOSEnvMock(),
): [typeof staticEnvironmentContext, StaticEnvironmentContext] {
  return [staticEnvironmentContext, staticEnvValue];
}

export const MockUserAgent = {
  Android: {
    6: {
      NexusTablet:
        'Mozilla/5.0 (Linux; Android 6.0.1; Nexus 10 Build/MOB31T) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/76.0.3788.1 Safari/537.36',
    },
    8: {
      PixelPhone:
        'Mozilla/5.0 (Linux; Android 8.0; Pixel 2 Build/OPD3.170816.012) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/76.0.3788.1 Mobile Safari/537.36',
    },
    8.1: {
      PixelPhone:
        'Mozilla/5.0 (Linux; Android 8.1; Pixel 2 Build/OPD3.170816.012) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/76.0.3788.1 Mobile Safari/537.36',
    },
    9: {
      FBWebView:
        'Mozilla/5.0 (Linux; Android 9.0; Pixel 2 Build/OPD3.170816.012) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/76.0.3788.1 Mobile Safari/537.36 [FB_IAB/Orca-Android;FBAV/221.1.0.15.157;]',
      PixelPhone:
        'Mozilla/5.0 (Linux; Android 9.0; Pixel 2 Build/OPD3.170816.012) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/76.0.3788.1 Mobile Safari/537.36',
    },
  },
  FireKindle:
    'Mozilla/5.0 (Linux; U; Android 4.2.2; fr-CA; Amazon Kindle Fire Build/JDQ39) AppleWebKit/534.30 (KHTML like Gecko) Version/4.0 Mobile Safari/534.30',
  Silk: 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Silk/44.1.54 like Chrome/44.0.2403.63 Safari/537.36',
  WindowsPhone:
    'Mozilla/5.0 (Mobile; Windows Phone 8.1; Android 4.0; ARM; Trident/7.0; Touch; rv:11.0; IEMobile/11.0; NOKIA; Lumia 635) like iPhone OS 7_0_3 Mac OS X AppleWebKit/537 (KHTML, like Gecko) Mobile Safari/537',
  iOS: {
    9: {
      iPad: 'Mozilla/5.0 (iPad; CPU OS 9_3 like Mac OS X) AppleWebKit/601.1.46 (KHTML, like Gecko) Version/9.0 Mobile/13E233 Safari/601.1',
      iPhone:
        'Mozilla/5.0 (iPhone; CPU iPhone OS 9_3 like Mac OS X) AppleWebKit/601.1.46 (KHTML, like Gecko) Version/9.0 Mobile/13E233 Safari/601.1',
    },
    10: {
      iPad: 'Mozilla/5.0 (iPad; CPU OS 10_3_2 like Mac OS X) AppleWebKit/603.2.4 (KHTML, like Gecko) Version/10.0 Mobile/14F90 Safari/602.1',
      iPhone:
        'Mozilla/5.0 (iPhone; CPU iPhone OS 10_0_1 like Mac OS X) AppleWebKit/602.1.50 (KHTML, like Gecko) Version/10.0 Mobile/14A403 Safari/602.1',
    },
    12: {
      FBWebView:
        'Mozilla/5.0 (iPhone; CPU iPhone OS 12_4_5 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Mobile/15E148 [FBAN/FBIOS;FBDV/iPhone7,2;FBMD/iPhone;FBSN/iOS;FBSV/12.4.5;FBSS/2;FBID/phone;FBLC/en_US;FBOP/5;FBCR/]',
      iPad: 'Mozilla/5.0 (iPad; CPU OS 12_2 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/12.1 Mobile/15E148 Safari/604.1',
      iPhone:
        'Mozilla/5.0 (iPhone; CPU iPhone OS 12_1_3 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/12.0 Mobile/15E148 Safari/604.1',
    },
    13: {
      FBWebView:
        'Mozilla/5.0 (iPhone; CPU iPhone OS 13_3_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Mobile/15E148 [FBAN/FBIOS;FBDV/iPhone11,8;FBMD/iPhone;FBSN/iOS;FBSV/13.3.1;FBSS/2;FBID/phone;FBLC/en_US;FBOP/5;FBCR/]',
    },
  },
  macOS: {
    Chrome:
      'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_14_3) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/71.0.3578.98 Safari/537.36',
    Firefox:
      'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.14; rv:63.0) Gecko/20100101 Firefox/63.0',
    Safari:
      'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_13_6) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/12.2 Safari/605.1.15',
  },
};

type MockEnvironmentContext = Omit<
  StaticEnvironmentContext,
  'releaseClientInfo'
>;

export function getAndroidEnvMock(): MockEnvironmentContext {
  return {
    client: {
      agentInfo: getAgentInfo(MockUserAgent.Android[8].PixelPhone),
      deviceID: 'android-chrome-device',
      netInfo: {
        bandwidth: 10,
        bandwidth_max: 15,
        effective_mobile_connection_type:
          EffectiveConnectionType.FourthGeneration,
        mobile_connection_type: ConnectionType.Cellular,
        round_trip_time: 50,
      },
      sessionID: 'android-chrome-session',
    },
    common: mockCommonDevelopmentEnvironment(),
  };
}

export function getiOSEnvMock(): MockEnvironmentContext {
  return {
    client: {
      agentInfo: getAgentInfo(MockUserAgent.iOS['12'].iPhone),
      deviceID: 'ios-safari-device',
      netInfo: {
        bandwidth: undefined,
        bandwidth_max: undefined,
        effective_mobile_connection_type: undefined,
        mobile_connection_type: undefined,
        round_trip_time: undefined,
      },
      sessionID: 'ios-safari-session',
    },
    common: mockCommonDevelopmentEnvironment(),
  };
}

export function getSilkOSEnvMock(): MockEnvironmentContext {
  return {
    client: {
      agentInfo: getAgentInfo(MockUserAgent.Silk),
      deviceID: 'silk-fire-os-device',
      netInfo: {
        bandwidth: undefined,
        bandwidth_max: undefined,
        effective_mobile_connection_type: undefined,
        mobile_connection_type: undefined,
        round_trip_time: undefined,
      },
      sessionID: 'silk-fire-session',
    },
    common: mockCommonDevelopmentEnvironment(),
  };
}

export function getWindowsPhoneEnvMock(): MockEnvironmentContext {
  return {
    client: {
      agentInfo: getAgentInfo(MockUserAgent.WindowsPhone),
      deviceID: 'windows-phone-device',
      netInfo: {
        bandwidth: undefined,
        bandwidth_max: undefined,
        effective_mobile_connection_type: undefined,
        mobile_connection_type: undefined,
        round_trip_time: undefined,
      },
      sessionID: 'windows-phone-session',
    },
    common: mockCommonDevelopmentEnvironment(),
  };
}

export function getOtherEnvMock(
  opts: {
    platform?: Platform;
    ua?: string;
  } = {},
): MockEnvironmentContext {
  const ua = opts.ua ?? 'Unknown User Agent';
  const platform = opts.platform ?? Platform.Unknown;
  return {
    client: {
      agentInfo: getAgentInfo(ua),
      deviceID: 'other-device',
      netInfo: {
        bandwidth: undefined,
        bandwidth_max: undefined,
        effective_mobile_connection_type: undefined,
        mobile_connection_type: undefined,
        round_trip_time: undefined,
      },
      sessionID: 'other-session',
    },
    common: mockCommonDevelopmentEnvironment(platform),
  };
}

export function mockCommonProductionEnvironment(
  platform = Platform.Unknown,
): CommonEnvironment {
  return {
    appEnvironment: 'production',
    appVersion: 'mock.0',
    clientApp: 'mock-app',
    language: 'en-US',
    platform,
  };
}

export function mockCommonDevelopmentEnvironment(
  platform = Platform.Unknown,
): CommonEnvironment {
  return {
    appEnvironment: 'development',
    appVersion: 'mock.0',
    clientApp: 'mock-app',
    language: 'en-US',
    platform,
  };
}
