import Bowser from 'bowser';
import { isBrowser } from 'tachyon-utils-stdlib';
import { BOWSER_USER_AGENT_FALLBACK } from 'tachyon-utils-twitch';

/**
 * Alias to simplify consumption.
 */
export type AgentInfo = Bowser.Parser.Parser;

export function getAgentInfo(userAgent: string): AgentInfo {
  return Bowser.getParser(userAgent || BOWSER_USER_AGENT_FALLBACK);
}

export function isIOS(agentInfo: AgentInfo): boolean {
  return agentInfo.getOSName() === 'iOS';
}

export function isAndroidOS(agentInfo: AgentInfo): boolean {
  return agentInfo.getOSName() === 'Android';
}

// TODO: clarify whether this is Kindle Fire or Silk or both
/**
 * Detects Fire tablets newer than 2015 based on 'silk' UA regex that Bowser uses.
 * See:
 * https://developer.amazon.com/docs/fire-tablets/ft-user-agent-strings.html
 */
export function isFireOS(agentInfo: AgentInfo): boolean {
  // Bowser doesn't detect Fire at the OS level, so we revert to using platform.
  // https://github.com/lancedikson/bowser/blob/master/src/parser-platforms.js
  // We do the model check below since we want parity with our legacy regex that
  // only cared about the presence of "silk"
  // const platform = agentInfo.getPlatform();
  // return platform.type === 'tablet' && platform.vendor === 'Amazon' && platform.model !== 'Kindle Fire HD 7';

  return /\bSilk\b/.test(agentInfo.getUA());
}

export function isMobileWeb(agentInfo: AgentInfo): boolean {
  const platform = agentInfo.getPlatformType();

  return ['tablet', 'mobile'].includes(platform);
}

const amazonEchoShowDeviceCode = 'AEO';

// Helper fn to fix https://jira.xarth.tv/browse/EMP-4026
export function isAmazonEchoShowDevice(agentInfo: AgentInfo): boolean {
  return (
    isBrowser() &&
    agentInfo.getBrowser().name === Bowser.BROWSER_MAP.amazon_silk &&
    window.navigator.userAgent.includes(amazonEchoShowDeviceCode)
  );
}

/**
 * These are the rules to check the dom for if we have access to it
 */
const webViewIdentifiers = [
  // webView object exists
  function (w: Window) {
    return 'webView' in w;
  },

  // Android object exists
  function (w: Window) {
    return 'Android' in w;
  },

  // deviceready event
  function (w: Window) {
    return w.document && 'ondeviceready' in w.document;
  },

  // explicitely states it is standalone
  function (w: Window) {
    return w.navigator && 'standalone' in w.navigator;
  },

  // can notifiy a parent app
  function (w: Window) {
    return w.external && 'notify' in w.external;
  },

  // Ti object exists (likely twitter detection?)
  function (w: Window) {
    return 'Ti' in w;
  },

  // _cordovaNative object exists
  function (w: Window) {
    return '_cordovaNative' in w;
  },

  // served as file (but false positive if a Web page is loaded from the filesystem)
  function (w: Window) {
    return w.location.href.indexOf('file:') === 0;
  },
];

// hard coded user agent partials from common user agents for FB, instagram & twitter (via: https://developers.whatismybrowser.com/useragents/)
const userAgentRules = [
  '[FB_IAB/',
  '[FBAN/FBIOS',
  'Twitter for iPhone',
  ' Instagram ',
];

/**
 * Takes in the agentInfo and determines if this is *likely* to be a WebView.
 * This works well for iOS but is flakey for Android.
 */
export function isLikelyWebView(agentInfo: AgentInfo): boolean {
  const userAgent = agentInfo.getUA();

  // if we're not in the browser we can skip this entirely
  if (!isBrowser()) {
    return false;
  }

  // check for non-W3C properties injected by wrappers
  if (webViewIdentifiers.some((rule) => rule(window))) {
    return true;
  }
  // mostly works for iOS - check userAgent for UIWebView
  if (
    userAgent.indexOf(' Mobile/') > 0 &&
    userAgent.indexOf(' Safari/') === -1
  ) {
    return true;
  }

  // test direct hard-coded fb, twitter & instagram identifiers
  if (userAgentRules.some((testString) => userAgent.includes(testString))) {
    return true;
  }
  return false;
}
