import { isBrowser } from 'tachyon-utils-stdlib';
import { MockUserAgent } from '../../test-mocks';
import {
  getAgentInfo,
  isAmazonEchoShowDevice,
  isAndroidOS,
  isFireOS,
  isIOS,
  isLikelyWebView,
  isMobileWeb,
} from '.';

interface IsMobileWebTestCase {
  UA: string;
  expected: boolean;
}

jest.mock('tachyon-utils-stdlib', () => ({
  ...jest.requireActual('tachyon-utils-stdlib'),
  isBrowser: jest.fn(() => true),
}));
const mockIsBrowser = isBrowser as jest.Mock;

describe('Agent Helpers', () => {
  describe(getAgentInfo, () => {
    it('returns an object', () => {
      expect(getAgentInfo(window.navigator.userAgent)).not.toBeFalsy();
    });

    it('does not choke on an empty UA', () => {
      expect(() => getAgentInfo('')).not.toThrow();
    });
  });

  describe(isIOS, () => {
    it('identifies iOS correctly', () => {
      expect(
        isIOS(getAgentInfo(MockUserAgent.Android['8'].PixelPhone)),
      ).toEqual(false);
      expect(isIOS(getAgentInfo(MockUserAgent.iOS['9'].iPhone))).toEqual(true);
    });
  });

  describe(isAndroidOS, () => {
    it('identifies Android correctly', () => {
      expect(
        isAndroidOS(getAgentInfo(MockUserAgent.Android['8'].PixelPhone)),
      ).toEqual(true);
      expect(isAndroidOS(getAgentInfo(MockUserAgent.iOS['9'].iPhone))).toEqual(
        false,
      );
    });
  });

  describe(isFireOS, () => {
    it('identifies Silk UAS as Fire OS', () => {
      expect(isFireOS(getAgentInfo(MockUserAgent.Silk))).toEqual(true);
    });

    it('does not identify Kindle Fire UAs as Fire OS', () => {
      expect(isFireOS(getAgentInfo(MockUserAgent.FireKindle))).toEqual(false);
    });
  });

  describe(isAmazonEchoShowDevice, () => {
    afterEach(() => {
      delete (window as any).navigator;
    });

    it('returns false when not in browser', () => {
      mockIsBrowser.mockImplementationOnce(() => false);
      expect(
        isAmazonEchoShowDevice(getAgentInfo(MockUserAgent.iOS['9'].iPhone)),
      ).toEqual(false);
    });

    it.each`
      UA
      ${'Mozilla/5.0 (Linux; Android 5.1.1; AEOKN Build/LVY48F) AppleWebKit/537.36 (KHTML, like Gecko) Silk/68.3.9999 like Chrome/68.0.3440.85 Safari/537.36'}
      ${'Mozilla/5.0 (Linux; Android 5.1.1; AEOBP Build/LVY48F) AppleWebKit/537.36 (KHTML, like Gecko) Silk/68.3.9999 like Chrome/68.0.3440.85 Safari/537.36'}
      ${'Mozilla/5.0 (Linux; Android 7.1.2; AEOCH) AppleWebKit/537.36 (KHTML, like Gecko) Silk/72.1.37 like Chrome/72.0.3626.0 Mobile Safari/537.36'}
      ${'Mozilla/5.0 (Linux; Android 7.1.2; AEOCW) AppleWebKit/537.36 (KHTML, like Gecko) Silk/78.0.9999 like Chrome/78.0.3904.0 Safari/537.36'}
      ${'Mozilla/5.0 (Linux; Android 9; AEOTA) AppleWebKit/537.36 (KHTML, like Gecko) Silk/85.0.9999 like Chrome/85.0.4183.0 Safari/537.36'}
      ${'Mozilla/5.0 (Linux; Android 9; AEOAT) AppleWebKit/537.36 (KHTML, like Gecko) Silk/90.1.94 like Chrome/90.0.4430.91 Safari/537.36'}
      ${'Mozzila/5.0 (Linux; Android 7.1.2; AEOCN) AppleWebkit/537.36 (KHTML, like Gecko) Silk/88.0.9999 Like Chrome/88.0.4324.0 Safari/537.36'}
    `(
      'returns true when in browser on Amazon Show Echo device with UA $UA',
      ({ UA }) => {
        (window as any).navigator = {
          userAgent: UA,
        };
        expect(isAmazonEchoShowDevice(getAgentInfo(UA))).toBe(true);
      },
    );

    it.each`
      type                         | UA
      ${'Android 6 Nexus Tablet'}  | ${MockUserAgent.Android[6].NexusTablet}
      ${'Android 8 Pixel Phone'}   | ${MockUserAgent.Android[8].PixelPhone}
      ${'Android 8.1 Pixel Phone'} | ${MockUserAgent.Android[8.1].PixelPhone}
      ${'Android 9 FB Web View'}   | ${MockUserAgent.Android[9].FBWebView}
      ${'Fire Kindle'}             | ${MockUserAgent.FireKindle}
      ${'Windows Phone'}           | ${MockUserAgent.WindowsPhone}
      ${'iOS 9 iPad'}              | ${MockUserAgent.iOS[9].iPad}
      ${'iOS 9 iPhone'}            | ${MockUserAgent.iOS[9].iPhone}
      ${'iOS 10 iPad'}             | ${MockUserAgent.iOS[10].iPad}
      ${'iOS 10 iPhone'}           | ${MockUserAgent.iOS[10].iPhone}
      ${'iOS 12 FBWebView'}        | ${MockUserAgent.iOS[12].FBWebView}
      ${'iOS 12 iPad'}             | ${MockUserAgent.iOS[12].iPad}
      ${'iOS 12 iPhone'}           | ${MockUserAgent.iOS[12].iPhone}
      ${'iOS 13 FBWebView'}        | ${MockUserAgent.iOS[13].FBWebView}
      ${'macOS Chrome'}            | ${MockUserAgent.macOS.Chrome}
      ${'macOS Firefox'}           | ${MockUserAgent.macOS.Firefox}
      ${'macOS Safari'}            | ${MockUserAgent.macOS.Safari}
    `('returns false when in browser on $type with UA $UA', ({ UA }) => {
      (window as any).navigator = {
        userAgent: UA,
      };
      expect(isAmazonEchoShowDevice(getAgentInfo(UA))).toBe(false);
    });
  });

  describe(isMobileWeb, () => {
    it.each`
      UA                                        | expected
      ${MockUserAgent.Android['6'].NexusTablet} | ${true}
      ${MockUserAgent.iOS['12'].iPad}           | ${true}
      ${MockUserAgent.iOS['9'].iPhone}          | ${true}
      ${MockUserAgent.WindowsPhone}             | ${true}
      ${MockUserAgent.macOS.Safari}             | ${false}
    `(
      'for $UA it returns $expected',
      ({ UA, expected }: IsMobileWebTestCase) => {
        const parser = getAgentInfo(UA);
        expect(isMobileWeb(parser)).toEqual(expected);
      },
    );
  });

  describe(isLikelyWebView, () => {
    afterEach(() => {
      delete (window as any).webView;
    });
    it('returns false for empty UA', () => {
      const parser = getAgentInfo('');
      expect(isLikelyWebView(parser)).toBe(false);
    });

    it('returns true when any of the domRules are hit', () => {
      const parser = getAgentInfo(MockUserAgent.Android['8'].PixelPhone);
      // to keep typescript happy
      (window as any).webView = true;
      expect(isLikelyWebView(parser)).toBe(true);
    });

    it.each`
      UA                                        | expected
      ${MockUserAgent.Android['6'].NexusTablet} | ${false}
      ${MockUserAgent.iOS['12'].iPad}           | ${false}
      ${MockUserAgent.iOS['9'].iPhone}          | ${false}
      ${MockUserAgent.WindowsPhone}             | ${false}
      ${MockUserAgent.macOS.Safari}             | ${false}
      ${MockUserAgent.iOS['12'].FBWebView}      | ${true}
      ${MockUserAgent.iOS['13'].FBWebView}      | ${true}
      ${MockUserAgent.Android['9'].FBWebView}   | ${true}
    `(
      'for $UA it returns $expected',
      ({ UA, expected }: IsMobileWebTestCase) => {
        const parser = getAgentInfo(UA);
        expect(isLikelyWebView(parser)).toEqual(expected);
      },
    );
  });
});
