import type { ComponentType, FC } from 'react';
import { getComponentName } from 'tachyon-utils-react';
import type { StaticEnvironmentContext } from '../EnvironmentRoot';
import { useStaticEnvironment } from '../useStaticEnvironment';

/**
 * Props for mixing into components wrapped by withStaticEnvironment.
 */
export interface StaticEnvironmentProps {
  staticEnv: StaticEnvironmentContext;
}

/**
 * withStaticEnvironment is a HOC that provides the staticEnv prop, containing
 * common and client-specific environment data to the wrapped component. If used
 * in an SSR context, the client environment will be filled with empty values.
 *
 * @param Component A component which needs access to static environment data.
 */
export function withStaticEnvironment<P extends StaticEnvironmentProps>(
  Comp: ComponentType<P>,
): FC<Omit<P, keyof StaticEnvironmentProps>> {
  const Environment: FC<Omit<P, keyof StaticEnvironmentProps>> = (props) => {
    const staticEnv = useStaticEnvironment();
    const newProps = {
      ...props,
      staticEnv,
    } as P;

    return <Comp {...newProps} />;
  };
  Environment.displayName = `StaticEnvironment(${getComponentName(Comp)})`;
  return Environment;
}
