function fixPrefix(fixer, identifierNode) {
  // fix improperly cased prefix
  if (identifierNode.name.toLowerCase().startsWith('sc')) {
    const [start] = identifierNode.range;
    return fixer.replaceTextRange([start, start + 2], 'Sc');
  }

  // fix missing prefix
  return fixer.insertTextBefore(identifierNode, 'Sc');
}

module.exports = {
  create: (context) => ({
    VariableDeclaration: (node) => {
      node.declarations.forEach((declaration) => {
        const { id, init } = declaration;
        if (
          init &&
          init.type === 'TaggedTemplateExpression' &&
          init.tag.object &&
          init.tag.object.name === 'styled' &&
          !id.name.startsWith('Sc')
        ) {
          // references includes the variable declaration itself also
          const { references } = context.getDeclaredVariables(declaration)[0];

          // If exported and the only reference, assume this is an opaque
          // component intended for general use and not styles used locally
          // inside a component, and assume exports with in-file usages are for
          // testability.
          if (
            declaration.parent.type !== 'ExportNamedDeclaration' &&
            references.length === 1
          ) {
            return;
          }

          context.report({
            fix: (fixer) => {
              const fixes = [];

              references
                .map(({ identifier }) => identifier)
                .forEach((fixable) => {
                  // also fix the closing tag for JSX tags when not self-closing
                  if (
                    fixable.type === 'JSXIdentifier' &&
                    fixable.parent.parent.closingElement
                  ) {
                    fixes.push(
                      fixPrefix(
                        fixer,
                        fixable.parent.parent.closingElement.name,
                      ),
                    );
                  }

                  fixes.push(fixPrefix(fixer, fixable));
                });

              return fixes;
            },
            message: 'styled-components names should be prefixed with `Sc`',
            node,
          });
        }
      });
    },
  }),
};
