# Tachyon Event Reporter

Provides a library for reporting events to Spade as well as other data sinks,
like the console, or whatever adapter matches the interface. Pairs nicely with
[tachyon-event-tracker](../event-tracker/README.md).

Spade reporting features include:

- Event batching
- [Beacon](https://developer.mozilla.org/en-US/docs/Web/API/Navigator/sendBeacon)
  use when possible to ensure event delivery even during unload

## Installation

```sh
$ yarn add tachyon-event-reporter
```

## How To Use

### Function Interface

```tsx
import {
  createConsoleReportEvent,
  createMultiReportEvent,
  createSpadeReportEvent,
} from 'tachyon-event-reporter';

const spadeURI = ... // can be pulled from dynamic settings or elsewhere

// in client/universal code somewhere
function createReportEvent() {
  return createMultiReportEvent([
    createConsoleReportEvent(),
    createSpadeReportEvent(spadeURI),
  ]);
}

// Silly example of reporting
const reportEvent = createReportEvent();
reportEvent({
  event: spadeReporter,
  properties: { spadeReporterStatus: 'initialized' },
});
```

### Component Interface

Example that turns on spade debugger. Switch `debugSpade` to `false` and set a
Spade URI to use production spade.

```tsx
import { EventReporterRoot, SPADE_ROTATOR_URI, useEventReporter } from 'tachyon-event-reporter';

const SomeComponent = () => {
  const { reportEvent } = useEventReporter();
  return (
    <button click={() => reportEvent(/* event stuff here */)} />,
  );
};

const root = (
  <EventReporterRoot
    console
    spade={{ spadeUrl: '...' // use SPADE_ROTATOR_URI to fetch the current Spade reporting URL }
  >
    <SomeComponent />
  </EventReporterRoot>
);
```

## Service Dependencies and Degradation

This section explains how tachyon-spade-reporter handles degraded availability
or outages with upstream services. The Tachyon Spade Reporter package makes use
of two services:

- The spade ingest URL provider (Tier 2?, unclear).
- Spade ingest service (Tier 2).

### Spade Ingest URL Provider

The package will attempt to periodically fetch the URL to which the client
should send spade events. If this attempt fails then it will fall back to the
last known-good URL that was fetched, if none exists it will fall back to the
default.

### Spade Ingest Service

This package follows a fire-and-forget strategy for spade events. This is in
accordance with advice from Data Science when the implementation was written. We
have a pending ticket to update the strategy to follow the retry logic the core
data science team now recommends.

## Flume Reporting

Flume is a reporting flow parallel to Spade reporting that uses application
services to populate all events to CloudWatch. Flume is a replacement for a
Kinesis sub-stream of Spade data, avoiding extra infra while resulting in the
same outcome: client event data available in CloudWatch streams.

## Usage

To activate it, pass the required config (`flumeUrl` and `allowedProperties`) to
the flume prop on `EventReporterRoot` (or create your own with
`createFlumeReportEvent`). To prevent accidentally logging PII, the
`allowedProperties` list is used to filter for specific event fields. Once
activated, the reporting framework will mirror all events to the flume endpoint
of your choosing.

```
+------------+  3. Process/Log  +------------+
| CloudWatch | <--------------- | App Server | <+
+------------+                  +------------+  |
                                |               |
                                | 1. SSR        | 2. POST /flume/handler/path
                                v               |
                                +------------+  |
                                | Web Client | -+
                                +------------+
```

### Serverside Configuration

A flume reporting endpoint must accept POST requests with support for
url-encoded form request bodies. The event payload will be under the `data` key,
and can simply be sent to the `logFlume` function to have the server properly
log the event in a CloudWatch-consumable way.

Once this route is configured and logging, events will be available in
CloudWatch logs under the normal `tachyon-logger` format where `message` is
equal to the event name (e.g. `pageview`) and `context` is equal to the filtered
event properties (e.g. `location`). In dev mode, the server logs will show these
messages as well, so that can be used as a reference to the event shapes.

#### In Tachyon

In a Tachyon app, it is recommended to use the `flumeHandler` function from
`tacyhon-page-utils` as a drop-in route handler:

```ts
// .../pages/api/my/flume/path.ts
export { flumeHandler as default } from 'tachyon-page-utils';
```

Furthermore, it is recommended to use an absolute path (e.g. `/api/flume`) in
the flume config for portability between environments.
