import type { FC } from 'react';
import { createContext, useMemo } from 'react';
import { useConst } from 'tachyon-utils-react';
import { isBrowser } from 'tachyon-utils-stdlib';
import type {
  FlumeReportEventOpts,
  SpadeReportEventOpts,
} from '../../reportEvent';
import {
  createConsoleReportEvent,
  createFlumeReportEvent,
  createMultiReportEvent,
  createSpadeReportEvent,
} from '../../reportEvent';
import type { ReportEvent } from '../../types';

export interface EventReporterContext {
  reportEvent: ReportEvent;
}

export const eventReporterContext = createContext<EventReporterContext>({
  reportEvent: createConsoleReportEvent(),
});

export type EventReporterRootProps = {
  /**
   * Controls whether events are reported to the console. Defaults to true.
   */
  console?: boolean | undefined;
  /**
   * Controls whether events are reported to a Flume endpoint, using the passed
   * in URL as the target and the allowed properties.
   */
  flume?: FlumeReportEventOpts | undefined;
  /**
   * Controls whether events are reported to Spade, using the passed in URL as
   * the target.
   */
  spade?: SpadeReportEventOpts | undefined;
};

export const EventReporterRoot: FC<EventReporterRootProps> = ({
  children,
  console = true,
  flume,
  spade,
}) => {
  const reportEvent = useConst(() => {
    const reportEvents: ReportEvent[] = [];

    if (console) {
      reportEvents.push(createConsoleReportEvent());
    }

    if (isBrowser()) {
      if (spade) {
        reportEvents.push(createSpadeReportEvent(spade));
      }

      if (flume) {
        reportEvents.push(createFlumeReportEvent(flume));
      }
    }

    return createMultiReportEvent(reportEvents);
  });

  const ctx = useMemo(() => ({ reportEvent }), [reportEvent]);

  return <eventReporterContext.Provider children={children} value={ctx} />;
};

EventReporterRoot.displayName = 'EventReporterRoot';
