import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { isBrowser } from 'tachyon-utils-stdlib';
import type {
  FlumeReportEventOpts,
  SpadeReportEventOpts,
} from '../../reportEvent';
import {
  createConsoleReportEvent,
  createFlumeReportEvent,
  createSpadeReportEvent,
} from '../../reportEvent';
import { EventReporterRoot } from '.';

jest.mock('tachyon-utils-stdlib', () => ({
  ...jest.requireActual('tachyon-utils-stdlib'),
  isBrowser: jest.fn(() => false),
}));
const mockIsBrowser = isBrowser as jest.Mock;

jest.mock('../../reportEvent', () => ({
  ...jest.requireActual('../../reportEvent'),
  createConsoleReportEvent: jest.fn(),
  createFlumeReportEvent: jest.fn(),
  createSpadeReportEvent: jest.fn(),
}));
const mockCreateConsoleReportEvent = createConsoleReportEvent as jest.Mock;
const mockCreateFlumeReportEvent = createFlumeReportEvent as jest.Mock;
const mockCreateSpadeReportEvent = createSpadeReportEvent as jest.Mock;

describe(EventReporterRoot, () => {
  const setup = createShallowWrapperFactory(EventReporterRoot);

  const spade: SpadeReportEventOpts = {
    batchWindowMs: 100,
    spadeUrl: '//spade',
  };
  const flume: FlumeReportEventOpts = {
    allowedEvents: ['fomo'],
    allowedProperties: ['yolo'],
    batchWindowMs: 100,
    flumeUrl: '//flume',
  };

  it('creates a console reportEvent by default ', () => {
    setup();
    expect(mockCreateConsoleReportEvent).toHaveBeenCalled();
    expect(mockCreateFlumeReportEvent).not.toHaveBeenCalled();
    expect(mockCreateSpadeReportEvent).not.toHaveBeenCalled();
  });

  it('does not create a console reportEvent functions if specified ', () => {
    setup({ console: false });
    expect(mockCreateConsoleReportEvent).not.toHaveBeenCalled();
    expect(mockCreateFlumeReportEvent).not.toHaveBeenCalled();
    expect(mockCreateSpadeReportEvent).not.toHaveBeenCalled();
  });

  it('creates a Spade reportEvent if specified and in the browser', () => {
    mockIsBrowser.mockImplementationOnce(() => true);
    setup({ spade });
    expect(mockCreateConsoleReportEvent).toHaveBeenCalled();
    expect(mockCreateFlumeReportEvent).not.toHaveBeenCalled();
    expect(mockCreateSpadeReportEvent).toHaveBeenCalledWith(spade);
  });

  it('does not creates a Spade reportEvent if specified but not in the browser', () => {
    setup({ spade });
    expect(mockCreateConsoleReportEvent).toHaveBeenCalled();
    expect(mockCreateFlumeReportEvent).not.toHaveBeenCalled();
    expect(mockCreateSpadeReportEvent).not.toHaveBeenCalled();
  });

  it('creates a Flume reportEvent if specified and in the browser', () => {
    mockIsBrowser.mockImplementationOnce(() => true);
    setup({ flume });
    expect(mockCreateConsoleReportEvent).toHaveBeenCalled();
    expect(mockCreateFlumeReportEvent).toHaveBeenCalledWith(flume);
    expect(mockCreateSpadeReportEvent).not.toHaveBeenCalled();
  });

  it('does not creates a Flume reportEvent if specified but not in the browser', () => {
    setup({ flume });
    expect(mockCreateConsoleReportEvent).toHaveBeenCalled();
    expect(mockCreateFlumeReportEvent).not.toHaveBeenCalled();
    expect(mockCreateSpadeReportEvent).not.toHaveBeenCalled();
  });
});
