import { logger } from 'tachyon-logger';
import { unspadeifyEvents } from 'tachyon-utils-crypto';
import { logFlume } from '.';

jest.mock('tachyon-logger', () => ({ logger: { log: jest.fn() } }));
const mockLoggerLog = logger.log as jest.Mock;

jest.mock('tachyon-utils-crypto', () => ({ unspadeifyEvents: jest.fn() }));
const mockUnspadeifyEvents = unspadeifyEvents as jest.Mock;

describe(logFlume, () => {
  const flumeStr = 'test';

  it('does nothing if event key is not present', () => {
    mockUnspadeifyEvents.mockImplementationOnce(() => [{ fakeEvent: true }]);
    logFlume(flumeStr);

    expect(mockUnspadeifyEvents).toHaveBeenCalledWith(flumeStr);
    expect(mockLoggerLog).not.toHaveBeenCalled();
  });

  it('logs event if event key is present', () => {
    const event = 'realEvent';
    const properties = { realEvent: true };
    mockUnspadeifyEvents.mockImplementationOnce(() => [{ event, properties }]);
    logFlume(flumeStr);

    expect(mockUnspadeifyEvents).toHaveBeenCalledWith(flumeStr);
    expect(mockLoggerLog).toHaveBeenCalledTimes(1);
    expect(mockLoggerLog).toHaveBeenCalledWith(
      expect.objectContaining({
        context: properties,
        message: event,
      }),
    );
  });

  it('logs multiple valid events', () => {
    const event = 'realEvent';
    const properties = { realEvent: true };
    mockUnspadeifyEvents.mockImplementationOnce(() => [
      { event, properties },
      { event, properties },
    ]);
    logFlume(flumeStr);

    expect(mockUnspadeifyEvents).toHaveBeenCalledWith(flumeStr);
    expect(mockLoggerLog).toHaveBeenCalledTimes(2);
    expect(mockLoggerLog).toHaveBeenCalledWith(
      expect.objectContaining({
        context: properties,
        message: event,
      }),
    );
  });

  it('logs decoding errors without throwing', () => {
    const errorStr = 'saw too big';
    mockUnspadeifyEvents.mockImplementationOnce(() => {
      throw new Error(errorStr);
    });
    expect(() => logFlume(flumeStr)).not.toThrow();

    expect(mockUnspadeifyEvents).toHaveBeenCalledWith(flumeStr);
    expect(mockLoggerLog).toHaveBeenCalledWith(
      expect.objectContaining({
        context: { error: expect.stringContaining(errorStr) },
        message: 'Error logging flume data',
      }),
    );
  });
});
