import type { TwitchEvent } from 'tachyon-type-library';
import type { FlumeifyEventOpts } from '../../flumeify';
import { flumeifyEvent } from '../../flumeify';
import type { ReportEvent } from '../../types';
import type { SpadeReportEventOpts } from '../createSpadeReportEvent';
import { createSpadeReportEvent } from '../createSpadeReportEvent';

export type FlumeReportEventOpts = Pick<
  FlumeifyEventOpts,
  'allowedProperties'
> &
  Pick<SpadeReportEventOpts, 'batchWindowMs'> & {
    allowedEvents: string[];
    flumeUrl: string;
  };

/**
 * Flume reportEvent reports events to a given Flume URL. Flume reporting is
 * identical to Spade reporting, except that it strips event properties that
 * aren't in the `allowedProperties` list to prevent proliferating PII.
 *
 * **Sending Extended Events**
 * Flume reportEvent defaults to events of type TwitchEvent. However, any
 * event that extends TwitchEvent can be sent through Flume reportEvent.
 */
export function createFlumeReportEvent<
  EventType extends TwitchEvent = TwitchEvent,
>({
  allowedEvents,
  allowedProperties,
  batchWindowMs,
  flumeUrl,
}: FlumeReportEventOpts): ReportEvent<EventType> {
  const spadeReportEvent = createSpadeReportEvent({
    batchWindowMs,
    spadeUrl: flumeUrl,
  });

  return (event) => {
    if (allowedEvents.includes(event.event)) {
      const flumeEvent = flumeifyEvent<EventType>({ allowedProperties, event });
      spadeReportEvent(flumeEvent);
    }
  };
}
