import type { TwitchEvent } from 'tachyon-type-library';
import type { ReportEvent } from '../../types';
import { sendEvents } from './sendEvents';

export type SpadeReportEventOpts = {
  /**
   * The size of an event batching window in milliseconds. This means that
   * events will be held for up to this many milliseconds waiting for more
   * events so that they can sent in single groups to decrease local resource
   * contention. Setting this to 0 disables batching and causes all events to be
   * sent individually.
   */
  batchWindowMs?: number | undefined;
  spadeUrl: string;
};

/**
 * Spade reportEvent reports events to a given Spade URL, performing the
 * necessary transformations and batching per the passed-in window size
 * (defaults to 1000 milliseconds).
 *
 * **Sending Extended Events**
 * Spade reportEvent defaults to events of type TwitchEvent. However, any
 * event that extends TwitchEvent can be sent through Spade reportEvent.
 */
export function createSpadeReportEvent<
  EventType extends TwitchEvent = TwitchEvent,
>({
  batchWindowMs = 1000,
  spadeUrl,
}: SpadeReportEventOpts): ReportEvent<EventType> {
  let eventQueue: EventType[] = [];
  let timeoutHandle: number | null = null;

  function sendEventsAndClearQueue(useBeacon = false): void {
    if (eventQueue.length > 0) {
      const events = eventQueue;
      eventQueue = [];
      sendEvents({ events, spadeUrl, useBeacon });
    }
  }

  // We attempt to use beacon only for reporting during background or unloading
  // of the app, which is its intended use-case. It's not meant as a general
  // purpose low-cost reporting structure and in fact has call and size limits
  // in some browsers.
  function sendEventsViaBeacon(): void {
    sendEventsAndClearQueue(true);
  }

  // use visibilitychange and pagehide to support mobile browsers
  window.addEventListener('visibilitychange', sendEventsViaBeacon);
  window.addEventListener('pagehide', sendEventsViaBeacon);
  // attempt beforeunload and unload as last resort
  window.addEventListener('beforeunload', sendEventsViaBeacon);
  window.addEventListener('unload', sendEventsViaBeacon);

  return (event) => {
    if (batchWindowMs === 0) {
      sendEvents({ events: [event], spadeUrl });
    }

    eventQueue.push(event);

    if (!timeoutHandle) {
      const sendFunc = () => {
        sendEventsAndClearQueue();
        timeoutHandle = null;
      };

      timeoutHandle = window.setTimeout(sendFunc, batchWindowMs);
    }
  };
}
