import type { TwitchEvent } from 'tachyon-type-library';
import { spadeifyEvents } from 'tachyon-utils-crypto';

export type SpadeReportEventOpts = {
  batchWindowMs: number;
  spadeUrl: string;
};

type SendEventsOps<EventType extends TwitchEvent = TwitchEvent> = {
  events: EventType[];
  spadeUrl: string;
  useBeacon?: boolean;
};

const DATA_CONTENT_TYPE = 'application/x-www-form-urlencoded; charset=UTF-8';

export function sendEvents<EventType extends TwitchEvent = TwitchEvent>({
  events,
  spadeUrl,
  useBeacon,
}: SendEventsOps<EventType>): void {
  const body = `data=${spadeifyEvents(events)}`;

  // Ideally the Beacon API will always be supported because the consumer
  // polyfills this for older browsers. If not, we fallback to fetch and hope it
  // works (e.g. if we get to unload, the browser might reap the request).
  if (useBeacon && 'sendBeacon' in navigator) {
    const blob = new Blob([body], {
      type: DATA_CONTENT_TYPE,
    });

    navigator.sendBeacon(spadeUrl, blob);
    return;
  }

  const payload: RequestInit = {
    body,
    headers: {
      'Content-Type': DATA_CONTENT_TYPE,
    },
    method: 'POST',
    mode: 'no-cors',
  };

  // attempt to report with a single retry, but fail silently
  fetch(spadeUrl, payload)
    .catch(() => fetch(spadeUrl, payload))
    .catch(() => undefined);
}
