import { spadeifyEvents } from 'tachyon-utils-crypto';
import { sendEvents } from '.';

jest.mock('tachyon-utils-crypto', () => ({
  spadeifyEvents: jest.fn(),
}));
const mockSpadeifyEvents = spadeifyEvents as jest.Mock;

describe(sendEvents, () => {
  beforeEach(() => {
    fetchMock.resetMocks();
  });

  const spadeUrl = 'https://spade.trowel.shovel.hoe.net';
  const events = [
    { event: 'foo1', properties: { iam: 'data1' } },
    { event: 'foo2', properties: { iam: 'data2' } },
  ];
  const base64Payload = 'base64';
  mockSpadeifyEvents.mockImplementation(() => base64Payload);

  it('reports an event using fetch by default', () => {
    sendEvents({ events, spadeUrl });
    expect(mockSpadeifyEvents).toHaveBeenCalledWith(events);
    expect(fetch).toHaveBeenCalledWith(spadeUrl, {
      body: `data=${base64Payload}`,
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
      },
      method: 'POST',
      mode: 'no-cors',
    });
  });

  describe('Beacon based reporting', () => {
    const realSendBeacon = window.navigator.sendBeacon;

    afterEach(() => {
      window.navigator.sendBeacon = realSendBeacon;
    });

    it('reports an event using a beacon when that option is set and the browser supports it', () => {
      window.navigator.sendBeacon = jest.fn();

      sendEvents({ events, spadeUrl, useBeacon: true });

      expect(mockSpadeifyEvents).toHaveBeenCalledWith(events);
      expect(window.navigator.sendBeacon).toHaveBeenCalledTimes(1);
      const [spadeURI, blob] = (window.navigator.sendBeacon as jest.Mock).mock
        .calls[0];

      expect(spadeURI).toEqual(spadeUrl);
      // Blobs can't be deep equaled, so lets at least make sure the arg is one
      expect(blob).toBeInstanceOf(Blob);
    });

    it('falls back to using fetch when the beacon option is set and the browser does not support it', () => {
      // @ts-expect-error: tests
      delete window.navigator.sendBeacon;

      sendEvents({ events, spadeUrl, useBeacon: true });

      expect(mockSpadeifyEvents).toHaveBeenCalledWith(events);
      expect(fetch).toHaveBeenCalledWith(spadeUrl, {
        body: `data=${base64Payload}`,
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
        },
        method: 'POST',
        mode: 'no-cors',
      });
    });
  });
});
