import { sendEvents } from './sendEvents';
import type { SpadeReportEventOpts } from '.';
import { createSpadeReportEvent } from '.';

jest.mock('./sendEvents', () => ({
  sendEvents: jest.fn().mockResolvedValue(undefined),
}));
const mockSendEvents = sendEvents as jest.Mock;

describe(createSpadeReportEvent, () => {
  const realAddEventListener = window.addEventListener;
  const mockAddEventListener = jest.fn();
  window.addEventListener = mockAddEventListener;

  afterAll(() => {
    window.addEventListener = realAddEventListener;
  });

  const opts: Required<SpadeReportEventOpts> = {
    batchWindowMs: 10,
    spadeUrl: '//spade',
  };
  const event = { event: 'foo1', properties: { iam: 'data1' } };

  it('registers abandonment listeners', () => {
    const report = createSpadeReportEvent(opts);
    report(event);

    expect(window.addEventListener).toHaveBeenCalledTimes(4);
    expect(window.addEventListener).toHaveBeenCalledWith(
      'visibilitychange',
      expect.any(Function),
    );
    expect(window.addEventListener).toHaveBeenCalledWith(
      'pagehide',
      expect.any(Function),
    );
    expect(window.addEventListener).toHaveBeenCalledWith(
      'beforeunload',
      expect.any(Function),
    );
    expect(window.addEventListener).toHaveBeenCalledWith(
      'unload',
      expect.any(Function),
    );
  });

  describe('batching', () => {
    it('creates a batching window on first invocation', () => {
      const report = createSpadeReportEvent(opts);
      report(event);

      expect(mockSendEvents).not.toHaveBeenCalled();

      jest.advanceTimersByTime(opts.batchWindowMs!);

      expect(mockSendEvents).toHaveBeenCalledWith({
        events: [event],
        spadeUrl: opts.spadeUrl,
        useBeacon: false,
      });
    });

    it('incorporates new events into an open batching window', () => {
      const report = createSpadeReportEvent(opts);
      report(event);

      expect(mockSendEvents).not.toHaveBeenCalled();

      jest.advanceTimersByTime(opts.batchWindowMs! - 1);
      report(event);
      expect(mockSendEvents).not.toHaveBeenCalled();

      jest.advanceTimersByTime(1);

      expect(mockSendEvents).toHaveBeenCalledWith({
        events: [event, event],
        spadeUrl: opts.spadeUrl,
        useBeacon: false,
      });
    });

    it('creates a new batching window for events that arrive outside of existing window', () => {
      const report = createSpadeReportEvent(opts);
      report(event);

      expect(mockSendEvents).not.toHaveBeenCalled();

      jest.advanceTimersByTime(opts.batchWindowMs!);
      report(event);

      expect(mockSendEvents).toHaveBeenCalledWith({
        events: [event],
        spadeUrl: opts.spadeUrl,
        useBeacon: false,
      });
      expect(mockSendEvents).toHaveBeenCalledTimes(1);

      jest.advanceTimersByTime(opts.batchWindowMs!);
      expect(mockSendEvents).toHaveBeenCalledTimes(2);
    });

    it('does not batch when window size is 0', () => {
      const report = createSpadeReportEvent({ ...opts, batchWindowMs: 0 });
      report(event);

      expect(mockSendEvents).toHaveBeenCalledWith({
        events: [event],
        spadeUrl: opts.spadeUrl,
      });
      expect(mockSendEvents).toHaveBeenCalledTimes(1);
    });
  });
});
