import type { FC, MutableRefObject } from 'react';
import { createContext, useCallback, useContext, useMemo, useRef } from 'react';
import { logger } from 'tachyon-logger';
import { useConst } from 'tachyon-utils-react';
import { isBrowser } from 'tachyon-utils-stdlib';
import { uniqueIDGenerator } from 'tachyon-utils-twitch';
import type { ProcessedEventData, RawEventData } from '../event-data';
import { EventType } from '../event-data';
import { processEvent } from '../processEvent';
import type {
  GetDynamicPropertiesOpts,
  GetStaticPropertiesOpts,
} from '../utils';
import { getStaticProperties } from '../utils';

/**
 * Internal function for passing events up via context
 */
export type OnRawEvent = (data: RawEventData) => void;

/**
 * Values that are typically used to capture where someone was referred/navigated
 * from before triggering an event.
 */
export type TwitchTracking = {
  /**
   * A subsection or type of medium. Off-site referrals will typically include
   * this as a "tt_content" URL query parameter.
   * ('share-button', 'promo-channels', 'message_share_link', etc)
   */
  content?: string | undefined;
  emailID?: string | undefined;
  /**
   * Top level description of where the traffic is coming from. Will either be
   * something external (email) or a prominent section that appears on multiple
   * sections of the site. Off-site referrals will typically include
   * this as a "tt_medium" URL query parameter.
   *
   * ('email', 'reddit', 'side-nav', etc)
   */
  medium?: string | undefined;
};

/**
 * Internal context provided by the EventTrackerRoot.
 */
export type EventTrackerContext = {
  /**
   * The current context within the tree of the app. This can be extended with
   * ExtendInteractionContext.
   */
  interactionMedium: string;
  /**
   * See: http://docs.sci.twitch.tv/events/pageview.html?search=content#location
   */
  location: string | undefined;
  /**
   * The callback for handling event tracking.
   */
  onEvent: OnRawEvent;
  twitchTracking: TwitchTracking;
};

// istanbul ignore next: trivial
export const eventTrackerContext = createContext<EventTrackerContext>({
  interactionMedium: 'test',
  location: undefined,
  onEvent: () => undefined,
  twitchTracking: {},
});

// for internal use only
export function useEventTrackerContext(): EventTrackerContext {
  return useContext(eventTrackerContext);
}

/**
 * Props for EventTrackerRoot.
 */
export type EventTrackerRootProps = Partial<
  Pick<GetDynamicPropertiesOpts, 'getUserTracking'>
> & {
  /**
   * Base interaction medium for this app's tracked interactions.
   */
  interactionMedium?: string;
  /**
   * See: http://docs.sci.twitch.tv/events/pageview.html?search=content#location
   */
  location?: string;
  /**
   * Function provided to the event handling system for consuming any events
   * generated. For instance, this could be an event reporting function.
   */
  onEvent: (data: ProcessedEventData) => void;
  /**
   * Data for generating the static properties attached to all events.
   */
  staticProperties: GetStaticPropertiesOpts;
  /**
   * Referral tracking parameters that a consumer should provide for certain types
   * of events that support the fields.
   */
  twitchTracking?: TwitchTracking;
};

type PageReference = {
  location: MutableRefObject<string | undefined>;
  onLocationChange: (location: EventTrackerRootProps['location']) => void;
  pageSessionId: MutableRefObject<string | undefined>;
};

function usePageReference(
  location: EventTrackerRootProps['location'],
): PageReference {
  const locationRef = useRef(location);
  const pageSessionId = useRef<string | undefined>();
  if (location && locationRef.current !== location) {
    pageSessionId.current = uniqueIDGenerator();
    locationRef.current = location;
  }

  const onLocationChange = useCallback(
    (loc: EventTrackerRootProps['location']) => {
      locationRef.current = loc;
      pageSessionId.current = uniqueIDGenerator();
    },
    [],
  );

  return {
    location: locationRef,
    onLocationChange,
    pageSessionId,
  };
}

/**
 * EventTrackerRoot initializes an event tracking system for a React
 * application, and must be put in the React tree above all other tracking
 * functionality from this package. It takes a function for consuming the event
 * data produced, as well as other base configuration options:
 *
 * - Providing interactionMedium sets a base interactionMedium value for all
 * UI interaction tracking within the app.
 *
 * - Providing location sets the location for all events in the app. The
 * location can set either here on EventTrackerRoot or on Pageview objects in
 * the subtree below this, depending on the router you're using and how to best
 * work with it. If you set location here, it will trump any location data
 * data sent in pageview events, so you cannot use both of them together.
 *
 * The location value will be normalized to null from undefined, but other falsy
 * values (specifically empty string) will not be clobbered to give more control
 * to the user.
 */
export const EventTrackerRoot: FC<EventTrackerRootProps> = ({
  children,
  getUserTracking,
  interactionMedium,
  location: locationOpt,
  onEvent,
  staticProperties: staticPropertiesOpts,
  twitchTracking,
}) => {
  const staticProperties = useConst(() =>
    getStaticProperties(staticPropertiesOpts),
  );

  const { location, onLocationChange, pageSessionId } =
    usePageReference(locationOpt);

  const eventHandler = useCallback(
    (data: RawEventData): void => {
      // istanbul ignore next: trivial
      if (!isBrowser()) {
        logger.error({
          category: 'EventTrackerRoot',
          context: {
            event: data.event,
            location: location.current ?? 'Unknown',
          },
          message: 'onEvent was invoked in a serverside context',
          package: 'event-tracker',
        });

        return;
      }

      if (
        !location.current &&
        data.event === EventType.Pageview &&
        data.properties.location !== undefined
      ) {
        onLocationChange(data.properties.location);
      }

      onEvent(
        processEvent({
          data,
          getUserTracking,
          location: location.current,
          page_session_id: pageSessionId.current,
          staticProperties,
        }),
      );
    },
    [
      getUserTracking,
      location,
      onEvent,
      onLocationChange,
      pageSessionId,
      staticProperties,
    ],
  );

  const ctx = useMemo(
    () => ({
      interactionMedium: interactionMedium ?? '',
      location: locationOpt,
      onEvent: eventHandler,
      twitchTracking: twitchTracking ?? {},
    }),
    [interactionMedium, locationOpt, eventHandler, twitchTracking],
  );

  return <eventTrackerContext.Provider children={children} value={ctx} />;
};

EventTrackerRoot.displayName = 'EventTrackerRoot';
