import type { FC } from 'react';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { uniqueIDGenerator } from 'tachyon-utils-twitch';
import type { PageviewEventProperties } from '../event-data';
import { EventType } from '../event-data';
import { processEvent } from '../processEvent';
import type { GetStaticPropertiesOpts } from '../utils';
import type { EventTrackerContext } from '.';
import { EventTrackerRoot, eventTrackerContext } from '.';

jest.mock('tachyon-utils-twitch', () => ({
  ...jest.requireActual('tachyon-utils-twitch'),
  uniqueIDGenerator: jest.fn(),
}));

jest.mock('../utils', () => ({
  ...jest.requireActual('../utils'),
  getDynamicProperties: jest.fn(({ location }) => ({ location })),
  getStaticProperties: jest.fn((p) => p),
}));

const mockUniqueIDGenerator = uniqueIDGenerator as jest.Mock;

describe(EventTrackerRoot, () => {
  const CtxReader: FC<EventTrackerContext> = () => null;

  const setup = createMountWrapperFactory(EventTrackerRoot, () => ({
    children: (
      <eventTrackerContext.Consumer
        children={(ctx) => <CtxReader {...ctx} />}
      />
    ),
    onEvent: jest.fn(),
    staticProperties: {
      appVersion: 'appVersion',
    } as GetStaticPropertiesOpts,
  }));

  it('augments events with standard fields', () => {
    mockUniqueIDGenerator.mockImplementationOnce(() => 'foo');

    const { props, wrapper } = setup();
    wrapper.find(CtxReader).prop('onEvent')({
      event: EventType.Custom,
      properties: { bar: 'baz', event: 'Custom' },
    });

    expect(props.onEvent).toHaveBeenCalledWith(
      processEvent({
        data: {
          event: EventType.Custom,
          properties: { bar: 'baz', event: 'Custom' },
        },
        location: undefined,
        page_session_id: 'foo',
        staticProperties: props.staticProperties as any,
      }),
    );
  });

  it('uses a new location and page_session_id when the location changes', () => {
    mockUniqueIDGenerator
      .mockImplementationOnce(() => 'first-page-session-id')
      .mockImplementationOnce(() => 'second-page-session-id');

    const { props, wrapper } = setup({ location: 'some-location' });
    wrapper.find(CtxReader).prop('onEvent')({
      event: EventType.Custom,
      properties: { bar: 'baz', event: 'Custom' },
    });

    expect(props.onEvent).toHaveBeenCalledWith(
      processEvent({
        data: {
          event: EventType.Custom,
          properties: { bar: 'baz', event: 'Custom' },
        },
        location: props.location,
        page_session_id: 'first-page-session-id',
        staticProperties: props.staticProperties as any,
      }),
    );

    wrapper.setProps({ location: 'next-location' });
    wrapper.find(CtxReader).prop('onEvent')({
      event: EventType.Custom,
      properties: { bar: 'baz', event: 'Custom' },
    });

    expect(props.onEvent).toHaveBeenCalledWith(
      processEvent({
        data: {
          event: EventType.Custom,
          properties: { bar: 'baz', event: 'Custom' },
        },
        location: 'next-location',
        page_session_id: 'second-page-session-id',
        staticProperties: props.staticProperties as any,
      }),
    );
  });

  it('uses a pageview location value when the root has not provided one', () => {
    mockUniqueIDGenerator.mockImplementationOnce(() => 'foo');

    const { props, wrapper } = setup();
    wrapper.find(CtxReader).prop('onEvent')({
      event: EventType.Pageview,
      properties: { location: 'baz' } as PageviewEventProperties,
    });

    expect(props.onEvent).toHaveBeenCalledWith(
      processEvent({
        data: {
          event: EventType.Pageview,
          properties: {} as PageviewEventProperties,
        },
        location: 'baz',
        page_session_id: 'foo',
        staticProperties: props.staticProperties as any,
      }),
    );

    wrapper.find(CtxReader).prop('onEvent')({
      event: EventType.Custom,
      properties: { event: 'Custom' },
    });

    expect(props.onEvent).toHaveBeenCalledWith(
      processEvent({
        data: {
          event: EventType.Custom,
          properties: { event: 'Custom' },
        },
        location: 'baz',
        page_session_id: 'foo',
        staticProperties: props.staticProperties as any,
      }),
    );
  });
});
