import type { FC } from 'react';
import { useEffect } from 'react';
import { useEventTrackerContext } from '../EventTrackerRoot';
import type {
  EventTrackerVODType,
  PageviewEventProperties,
} from '../event-data';
import { EventType } from '../event-data';

export type PageviewProps = {
  channel?: string | undefined;
  channelID?: string | undefined;
  clientOffline?: boolean | undefined;
  content?: string | undefined;
  game?: string | undefined;
  isLive?: boolean | undefined;
  location?: string | undefined;
  medium?: string | undefined;
  vodID?: string | undefined;
  vodType?: EventTrackerVODType | undefined;
};

/**
 * Hook for tracking pageview events, for special cases when the `<Pageview />`
 * component cannot be used. See that component's docs for usage.
 */
export function usePageview(props: PageviewProps): void {
  const {
    location: ctxLocation,
    onEvent,
    twitchTracking,
  } = useEventTrackerContext();
  const location = props.location ?? ctxLocation;

  useEffect(() => {
    const properties: PageviewEventProperties = {
      channel: props.channel,
      channel_id: props.channelID ? Number(props.channelID) : undefined,
      client_offline: props.clientOffline,
      content: twitchTracking.content,
      email_id: twitchTracking.emailID,
      game: props.game?.toLowerCase(),
      is_live: props.isLive,
      location,
      medium: twitchTracking.medium,
      vod_id: props.vodID,
      vod_type: props.vodType,
    };

    onEvent({
      event: EventType.Pageview,
      properties,
    });
    // Only location changes should trigger this event
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [location]);
}

/**
 * The Pageview component is a way to declaratively track pageview events.
 * It accepts an optional location prop for managing the location of the
 * event tracking system. It also accepts the other values like channel or vodID
 * for various page-specific tracking values.
 *
 * If you're managing location with Pageview objects, it is recommended to have
 * them as high up in your tree and as early in a JSX block as possible in order
 * to prevent events reporting with the old location value. The event system
 * tries to properly leverage the React component lifecycle to avoid this as
 * much as possible, but it is something of which to be aware. Because of this
 * possible interaction between tree location and event reporting order,
 * EventTrackerRoot is the preferred way to manage location and Pageview is
 * provided for situation where that may not be possible or for more advanced
 * implementations (such as nested multiple Pageviews in a single react-router
 * render tree, with the lowest one winning).
 *
 * Since the EventTrackerRoot is the preferred method of managing location, if
 * both EventTrackerRoot and Pageview are reporting location values, any
 * location value set on this component will be disregarded and the value from
 * EventTrackerRoot will win.
 */
export const Pageview: FC<PageviewProps> = (props) => {
  usePageview(props);

  return null;
};

Pageview.displayName = 'Pageview';
